"use strict";
/********************************************************************************
 * Copyright (C) 2021 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MemoryEditableTableWidget = exports.EditableMemoryWidget = void 0;
const browser_1 = require("@theia/core/lib/browser");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const inversify_1 = require("@theia/core/shared/inversify");
const React = require("@theia/core/shared/react");
const memory_table_widget_1 = require("../memory-widget/memory-table-widget");
const Long = require("long");
const util_1 = require("../../common/util");
var EditableMemoryWidget;
(function (EditableMemoryWidget) {
    EditableMemoryWidget.ID = 'editable.memory.widget';
})(EditableMemoryWidget = exports.EditableMemoryWidget || (exports.EditableMemoryWidget = {}));
let MemoryEditableTableWidget = class MemoryEditableTableWidget extends memory_table_widget_1.MemoryTableWidget {
    constructor() {
        super(...arguments);
        this.pendingMemoryEdits = new Map();
        this.memoryEditsCompleted = new promise_util_1.Deferred();
        this.highlightedField = Long.fromInt(-1);
        this.doShowMoreMemoryBefore = false;
        this.doShowMoreMemoryAfter = false;
        this.handleClearEditClick = () => this.clearEdits();
        this.submitMemoryEdits = async () => {
            this.memoryEditsCompleted = new promise_util_1.Deferred();
            for (const edit of this.createUniqueEdits()) {
                try {
                    await this.memoryProvider.writeMemory(edit);
                }
                catch (e) {
                    console.log('Problem writing memory with arguments', edit, '\n', e);
                }
            }
            this.memoryEditsCompleted.resolve();
        };
        this.handleTableClick = (event) => {
            var _a, _b;
            const target = event.target;
            if ((_a = target.classList) === null || _a === void 0 ? void 0 : _a.contains('eight-bits')) {
                this.highlightedField = util_1.hexStrToUnsignedLong((_b = target.getAttribute('data-id')) !== null && _b !== void 0 ? _b : '-0x1');
                this.update();
                event.stopPropagation();
            }
        };
        // eslint-disable-next-line max-lines-per-function,complexity
        this.handleTableInput = (event) => {
            var _a, _b;
            if (this.highlightedField.lessThan(0)) {
                return;
            }
            const keyCode = (_a = browser_1.KeyCode.createKeyCode(event.nativeEvent).key) === null || _a === void 0 ? void 0 : _a.keyCode;
            const initialHighlight = this.highlightedField;
            const initialHighlightIndex = initialHighlight.subtract(this.memory.address);
            if (keyCode === browser_1.Key.TAB.keyCode) {
                return;
            }
            const arrayElementsPerRow = (this.options.byteSize / 8) * this.options.bytesPerGroup * this.options.groupsPerRow;
            const isAlreadyEdited = this.pendingMemoryEdits.has(this.highlightedField.toString());
            const oldValue = (_b = this.pendingMemoryEdits.get(initialHighlight.toString())) !== null && _b !== void 0 ? _b : this.memory.bytes[initialHighlightIndex.toInt()].toString(16).padStart(2, '0');
            let possibleNewHighlight = new Long(-1);
            let newValue = oldValue;
            switch (keyCode) {
                case browser_1.Key.ARROW_DOWN.keyCode:
                    possibleNewHighlight = initialHighlight.add(arrayElementsPerRow);
                    event.preventDefault();
                    event.stopPropagation();
                    break;
                case browser_1.Key.ARROW_UP.keyCode:
                    possibleNewHighlight = initialHighlight.greaterThan(arrayElementsPerRow) ? initialHighlight.subtract(arrayElementsPerRow) : possibleNewHighlight;
                    event.preventDefault();
                    event.stopPropagation();
                    break;
                case browser_1.Key.ARROW_RIGHT.keyCode:
                    possibleNewHighlight = initialHighlight.add(1);
                    event.preventDefault();
                    event.stopPropagation();
                    break;
                case browser_1.Key.ARROW_LEFT.keyCode:
                    possibleNewHighlight = initialHighlight.greaterThan(0) ? initialHighlight.subtract(1) : possibleNewHighlight;
                    break;
                case browser_1.Key.BACKSPACE.keyCode:
                    newValue = oldValue.slice(0, oldValue.length - 1);
                    break;
                case browser_1.Key.DELETE.keyCode:
                    newValue = '';
                    break;
                case browser_1.Key.ENTER.keyCode:
                    this.submitMemoryEdits();
                    break;
                case browser_1.Key.ESCAPE.keyCode:
                    if (isAlreadyEdited) {
                        this.clearEdits(this.highlightedField);
                    }
                    else {
                        this.clearEdits();
                    }
                    break;
                default: {
                    const keyValue = parseInt(browser_1.KeyCode.createKeyCode(event.nativeEvent).toString(), 16);
                    if (!Number.isNaN(keyValue)) {
                        newValue = isAlreadyEdited ? oldValue : '';
                        if (newValue.length < 2) {
                            newValue += keyValue.toString(16);
                        }
                    }
                }
            }
            if (this.isInBounds(possibleNewHighlight)) {
                this.highlightedField = possibleNewHighlight;
            }
            const valueWasChanged = newValue !== oldValue;
            if (valueWasChanged) {
                this.pendingMemoryEdits.set(this.highlightedField.toString(), newValue);
            }
            if (valueWasChanged || !this.highlightedField.equals(initialHighlight)) {
                this.update();
            }
        };
    }
    async init() {
        this.memoryEditsCompleted.resolve();
        await super.init();
    }
    resetModifiedValue(valueAddress) {
        const didChange = this.pendingMemoryEdits.delete(valueAddress.toString());
        if (didChange) {
            this.update();
        }
    }
    getState() {
        super.getState();
        if (!this.isInBounds(this.highlightedField)) {
            this.highlightedField = this.memory.address;
        }
    }
    async handleMemoryChange(newMemory) {
        await this.memoryEditsCompleted.promise;
        this.pendingMemoryEdits.clear();
        super.handleMemoryChange(newMemory);
    }
    areSameRegion(a, b) {
        return b !== undefined && a.address.equals(b.address) && a.bytes.length === b.bytes.length;
    }
    getTableFooter() {
        return (!!this.pendingMemoryEdits.size && (React.createElement("div", { className: 'memory-edit-button-container' },
            React.createElement("button", { className: 'theia-button secondary', onClick: this.handleClearEditClick, type: 'reset' }, "Clear Changes"),
            React.createElement("button", { className: 'theia-button main', onClick: this.submitMemoryEdits, type: 'submit' }, "Apply Changes"))));
    }
    getBitAttributes(arrayOffset, iteratee) {
        var _a, _b, _c;
        const attributes = super.getBitAttributes(arrayOffset, iteratee);
        const classNames = (_b = (_a = attributes.className) === null || _a === void 0 ? void 0 : _a.split(' ')) !== null && _b !== void 0 ? _b : [];
        const itemID = this.memory.address.add(arrayOffset);
        const isHighlight = itemID.equals(this.highlightedField);
        const isEditPending = this.pendingMemoryEdits.has(itemID.toString());
        const padder = isHighlight && isEditPending ? '\xa0' : '0'; // non-breaking space so it doesn't look like plain whitespace.
        const stringValue = ((_c = this.pendingMemoryEdits.get(itemID.toString())) !== null && _c !== void 0 ? _c : this.memory.bytes[arrayOffset].toString(16)).padStart(2, padder);
        if (!this.options.isFrozen) {
            if (isHighlight) {
                classNames.push('highlight');
            }
            if (isEditPending) {
                classNames.push('modified');
            }
        }
        return Object.assign(Object.assign({}, attributes), { className: classNames.join(' '), content: stringValue });
    }
    getHoverForChunk(span) {
        const addressAsString = span.getAttribute('data-id');
        if (addressAsString) {
            const address = util_1.hexStrToUnsignedLong(addressAsString);
            const { value } = this.composeByte(address, true);
            const { value: inMemory } = this.composeByte(address, false);
            const oldValue = this.previousBytes && this.composeByte(address, false, this.previousBytes).value;
            const decimal = parseInt(value, 16);
            const octal = decimal.toString(8).padStart(this.options.byteSize / 8, '0');
            const UTF8 = String.fromCharCode(decimal);
            const binary = this.getPaddedBinary(decimal);
            const toSend = { hex: value, octal, binary, decimal };
            if (UTF8) {
                toSend.UTF8 = UTF8;
            }
            if (inMemory !== value) {
                toSend['Current Value'] = inMemory;
            }
            if (oldValue !== undefined && oldValue !== value) {
                toSend['Previous Value'] = oldValue;
            }
            return toSend;
        }
        return undefined;
    }
    composeByte(addressPlusArrayOffset, usePendingEdits, dataSource = this.memory.bytes) {
        let value = '';
        const offset = addressPlusArrayOffset.subtract(this.memory.address);
        const chunksPerByte = this.options.byteSize / 8;
        const startingChunkIndex = offset.subtract(offset.modulo(chunksPerByte));
        const address = this.memory.address.add(startingChunkIndex.divide(chunksPerByte));
        for (let i = 0; i < chunksPerByte; i += 1) {
            const targetOffset = startingChunkIndex.add(i);
            const targetChunk = this.getFromMapOrArray(targetOffset, usePendingEdits, dataSource);
            value += targetChunk.padStart(2, '0');
        }
        return { address, value };
    }
    getFromMapOrArray(arrayOffset, usePendingEdits, dataSource = this.memory.bytes) {
        var _a, _b;
        let value = usePendingEdits ? this.pendingMemoryEdits.get(arrayOffset.add(this.memory.address).toString()) : undefined;
        if (value === undefined) {
            value = (_b = (_a = dataSource[arrayOffset.toInt()]) === null || _a === void 0 ? void 0 : _a.toString(16)) !== null && _b !== void 0 ? _b : '';
        }
        return value;
    }
    clearEdits(address) {
        if (typeof address === 'number') {
            this.pendingMemoryEdits.delete(address);
        }
        else {
            this.pendingMemoryEdits.clear();
        }
        this.update();
    }
    createUniqueEdits() {
        const addressesSubmitted = new Set();
        const edits = [];
        for (const k of this.pendingMemoryEdits.keys()) {
            const address = Long.fromString(k);
            const { address: addressToSend, value: valueToSend } = this.composeByte(address, true);
            const memoryReference = '0x' + addressToSend.toString(16);
            if (!addressesSubmitted.has(memoryReference)) {
                const data = Buffer.from(valueToSend, 'hex').toString('base64');
                edits.push({ memoryReference, data });
                addressesSubmitted.add(memoryReference);
            }
        }
        return edits;
    }
    getWrapperHandlers() {
        return this.options.isFrozen
            ? super.getWrapperHandlers()
            : {
                onClick: this.handleTableClick,
                onContextMenu: this.handleTableRightClick,
                onKeyDown: this.handleTableInput,
                onMouseMove: this.handleTableMouseMove,
            };
    }
    doHandleTableRightClick(event) {
        var _a, _b;
        const target = event.target;
        if ((_a = target.classList) === null || _a === void 0 ? void 0 : _a.contains('eight-bits')) {
            this.highlightedField = util_1.hexStrToUnsignedLong((_b = target.getAttribute('data-id')) !== null && _b !== void 0 ? _b : '-0x1');
        }
        super.doHandleTableRightClick(event);
    }
    isInBounds(candidateAddress) {
        const { address, bytes } = this.memory;
        return candidateAddress.greaterThanOrEqual(address) &&
            candidateAddress.lessThan(address.add(bytes.length));
    }
};
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MemoryEditableTableWidget.prototype, "init", null);
MemoryEditableTableWidget = __decorate([
    inversify_1.injectable()
], MemoryEditableTableWidget);
exports.MemoryEditableTableWidget = MemoryEditableTableWidget;
//# sourceMappingURL=memory-editable-table-widget.js.map