#******************************************************************************
#
#       Copyright:      2005-2010 Paul Obermeier (obermeier@tcl3d.org)
#
#                       See the file "Tcl3D_License.txt" for information on
#                       usage and redistribution of this file, and for a
#                       DISCLAIMER OF ALL WARRANTIES.
#
#       Module:         Tcl3D extension package
#       Filename:       img2pkg.tcl
#
#       Author:         Paul Obermeier
#
#       Description:    This script expects PNG image filenames or directory
#                       names as parameters and writes a namespaced Tcl package
#                       to stdout.
#                       If a directory name is given, all files with suffix
#                       ".png" are included in the generated package.
#
#                       Assume the package name is "::gaugeImg" and one of the
#                       PNG files is called "PwrdByTcl.png".
#                       The generated package then provides functions of the
#                       following form:
#                               proc ::gaugeImg::PwrdByTcl {} {
#                                       return <UUencoded PNG or GIF image data>
#                               }
#
#                       You may change the name of the namespace/package and
#                       it's version number by editing the marked lines below.
#
#                       Note: This program expects the file gaugeBase64.tcl 
#                       (from the Tcl library tcllib) to be in the same
#                       directory as this script file.
#
#******************************************************************************

if { $argc < 1 } {
    puts stderr "No input files given for img2pkg.tcl"
    exit 1
}

# Parse arguments and generate a list of all files to be converted.
set fileList {}
foreach val $argv {
    if { [file isdirectory $val] } {
        foreach f [lsort [glob -nocomplain -directory $val *.png]] {
            lappend fileList $f
        }
        foreach f [lsort [glob -nocomplain -directory $val *.gif]] {
            lappend fileList $f
        }
    } elseif { [file isfile $val] } {
        lappend fileList $val
    }
}


set packageName   "tcl3dgauges"
set namespaceName "::gaugeImg"
set versionNum    "0.5.0"
source [file join [file dirname [info script]] "gaugeBase64.tcl"]

fconfigure stdout -translation binary

# Write package header.
puts "package provide $packageName $versionNum"
puts ""

# Write namespace header.
puts "namespace eval $namespaceName {"
foreach fileName $fileList {
    set imgName  [file rootname [file tail $fileName]]
    if { [file readable $fileName] } {
        puts "    namespace export $imgName"
    }
}
puts "}"
puts ""

# Generate namespace procedures.
foreach fileName $fileList {
    set imgName  [file rootname [file tail $fileName]]
    set procName [format "%s::%s" $namespaceName $imgName]

    if { [file readable $fileName] } {
        # puts stderr "Generating proc $procName (from $fileName)"
        set fp [open $fileName r]
        fconfigure $fp -translation binary
        set imgData [read $fp]
        close $fp
        set uuImg [::gaugeBase64::encode $imgData]
        puts "proc $procName \{\} \{"
        puts "return \{"
        puts -nonewline $uuImg
        puts "\}"
        puts "\} \; \# End of proc $procName"
        puts ""
    } else {
        puts stderr "File $fileName does not exist."
        exit 1
    }
}

puts ""
exit 0
