/* 
    TowBowlTactics: Strategy game for X11

    Copyright (C) 2002 Pascal Bourut (toweld@rocketmail.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef NOSOUND

#include "SoundPlayer.h"

const int VOLUME_STEP=5;

SoundPlayer::SoundPlayer(){
	
	SDL_InitSubSystem(SDL_INIT_AUDIO);

	freq = 44100;//22050;
	format = AUDIO_S16; 
	channels = 2;
	buffers = 4096;

    audiofmt.format = format;
    audiofmt.freq = freq;
    audiofmt.channels = channels;
	if(Mix_OpenAudio(freq, format, 2, buffers)) {
		fprintf(stderr, "Unable to open audio %d %d %d !\n",freq,format, channels);
		audio_opened = false;
		return;
	}
	
	audio_opened = true;
	int i;

	for(i=0;i<NUM_MUSICS;i++)
		add_music(music_names[i],i);
	current_music = 0;
	

	for(i=0;i<NUM_SOUNDS;i++)
		add_sound(sound_names[i],i);
	enable = true;	
}

SoundPlayer::~SoundPlayer(){
	enable = false;
	remove_sound(0);
	
	remove_music(0);

	Mix_CloseAudio();
	SDL_QuitSubSystem(SDL_INIT_AUDIO);
}

SDL_AudioSpec SoundPlayer::get_audiospec(){
	return audiofmt;
}


int SoundPlayer::get_current_music(){
	return current_music;
}


void SoundPlayer::set_enabled(bool b){
	enable = b;
	if(!enable){
		for(int i=0;i<NUM_MUSICS;i++) stop_music(i);
		for(int c=0;c<=2;c++) stop_sound(c);
	}
}


void SoundPlayer::add_music(const char*filename,int i){
	if(audio_opened && i<NUM_MUSICS ) {	
		music[i] = SMPEG_new(filename,&info,0);
		if(info.has_audio){
			SMPEG_actualSpec(music[i], &audiofmt);
			Mix_HookMusic(SMPEG_playAudioSDL, music[i]);
			SMPEG_enableaudio(music[i], 1);
		}
		else {
			fprintf(stderr, "%s does not contains audio stream !\n", filename);
			music[i] = NULL; 
		}
	}
}


void SoundPlayer::remove_music(int i){
	if(audio_opened && i<NUM_MUSICS && music[i]) {
		SMPEG_stop(music[i]);
		Mix_HookMusic(NULL, NULL);
		SMPEG_delete(music[i]);
		music[i] = NULL;
	}	
}

void SoundPlayer::play_music(int i){
	if(!enable) return; 
	if(audio_opened && i<NUM_MUSICS && music[i]){
		SMPEG_play(music[i]);
		current_music = i;
	}	
}

void SoundPlayer::pause_music(int i){
	if(audio_opened && i<NUM_MUSICS && music[i]){
		SMPEG_pause(music[i]);
	}	
}

void SoundPlayer::toggle_play_pause(int i){
	if(audio_opened && i<NUM_MUSICS && music[i]){
		if(SMPEG_status(music[i])==SMPEG_PLAYING) pause_music(i);
		else play_music(i);
	}
}

void SoundPlayer::toggle_play_pause(int i, bool b){
	if(audio_opened && i<NUM_MUSICS && music[i]){
		if(b) play_music(i);
		else pause_music(i);
	}
}

void SoundPlayer::stop_music(int i){
	if(audio_opened && i<NUM_MUSICS && music[i]){
		SMPEG_stop(music[i]);
		SMPEG_rewind(music[i]);
	}
}


void SoundPlayer::add_sound(const char*filename,int i){
	if(audio_opened && i<NUM_SOUNDS) {
		sounds[i] = Mix_LoadWAV(filename);
		if(!sounds[i]) {
			fprintf(stderr, "Mix_LoadWAV: %s\n", Mix_GetError());
		}
	}
}

void SoundPlayer::remove_sound(int i){
	if(audio_opened && i<NUM_SOUNDS && sounds[i]) {
		sounds[i] = NULL;
	}
}

void SoundPlayer::play_sound(int i,int chan){
	if(!enable) return;
	if(!audio_opened) return; 
	Mix_PlayChannel(chan, sounds[i], chan);
}

void SoundPlayer::play_sound(int i){
	if(!enable) return;
	if(!audio_opened) return;
	Mix_PlayChannel(FX0_CHANNEL, sounds[i], 0);
}

void SoundPlayer::loop_sound(int i){
	if(!enable) return;
	if(!audio_opened) return;
	Mix_PlayChannel(FX0_CHANNEL, sounds[i], -1);
}

void SoundPlayer::stop_sound(){
	if(!audio_opened) return;
	for(int c=0;c<=2;c++) stop_sound(c);
}

void SoundPlayer::loop_sound(int i,int chan){
	if(!enable) return;
	if(!audio_opened) return;
	Mix_PlayChannel(chan, sounds[i], -1);
}

void SoundPlayer::stop_sound(int chan){
	if(!audio_opened) return;
	Mix_HaltChannel(chan);
}

void SoundPlayer::dec_volume(){
	if(!audio_opened) return;
	if(volume>VOLUME_STEP) volume-=VOLUME_STEP;
	set_volume(volume);
}

void SoundPlayer::inc_volume(){
	if(!audio_opened) return;
	if(volume<(100-VOLUME_STEP)) volume+=VOLUME_STEP;
	set_volume(volume);
}

void SoundPlayer::set_volume(int i){
	if(!audio_opened) return;
	if(i<0) volume = 0;
	else if(i>100) volume = 100;
	else volume = i;
	for(int j=0;j<NUM_MUSICS;j++)
		SMPEG_setvolume(music[j],volume);
}

#else
; 

#endif

