/********************************/
/* Scehematic component edition */
/********************************/

#include "fctsys.h"
#include "gr_basic.h"
#include "common.h"
#include "class_drawpanel.h"
#include "confirm.h"
#include "class_sch_screen.h"
#include "wxEeschemaStruct.h"

#include "general.h"
#include "protos.h"
#include "class_library.h"
#include "sch_component.h"


static void AbortMoveCmpField( EDA_DRAW_PANEL* Panel, wxDC* DC );
static void MoveCmpField( EDA_DRAW_PANEL* aPanel, wxDC* aDC, const wxPoint& aPosition,
                          bool aErase );


/******************************************************************************/
/* Prepare the displacement of the text being edited.
 */
/******************************************************************************/
void SCH_EDIT_FRAME::StartMoveCmpField( SCH_FIELD* aField, wxDC* DC )
{
    LIB_COMPONENT* Entry;

    SetCurrentField( aField );
    if( aField == NULL )
        return;

    if( aField->m_Text == wxEmptyString )
    {
        DisplayError( this, _( "No Field to move" ), 10 );
        return;
    }

    wxPoint        pos, newpos;
    SCH_COMPONENT* comp = (SCH_COMPONENT*) aField->GetParent();

    SAFE_DELETE( g_ItemToUndoCopy );
    g_ItemToUndoCopy = new SCH_COMPONENT( *comp );

    pos = comp->m_Pos;

    /* Positions are computed by the rotation/mirror transform. */
    newpos = aField->m_Pos - pos;

    newpos = comp->GetTransform().TransformCoordinate( newpos ) + pos;

    DrawPanel->CrossHairOff( DC );
    GetScreen()->SetCrossHairPosition( newpos );
    DrawPanel->MoveCursorToCrossHair();

    m_OldPos    = aField->m_Pos;
    m_Multiflag = 0;
    if( aField->m_FieldId == REFERENCE )
    {
        Entry = CMP_LIBRARY::FindLibraryComponent( comp->GetLibName() );

        if( Entry  != NULL )
        {
            if( Entry->GetPartCount() > 1 )
                m_Multiflag = 1;
        }
    }

    DrawPanel->m_endMouseCaptureCallback = AbortMoveCmpField;
    DrawPanel->m_mouseCaptureCallback = MoveCmpField;
    aField->m_Flags = IS_MOVED;

    DrawPanel->CrossHairOn( DC );
}


/*
 * Edit a field: text and size
*/
void SCH_EDIT_FRAME::EditComponentFieldText( SCH_FIELD* aField, wxDC* aDC )
{
    wxCHECK_RET( aField != NULL && aField->Type() == SCH_FIELD_T,
                 wxT( "Invalid schemaitic field type. " ) );

    int            fieldNdx, flag;
    SCH_COMPONENT* component = (SCH_COMPONENT*) aField->GetParent();

    wxCHECK_RET( component != NULL && component->Type() == SCH_COMPONENT_T,
                 wxT( "Invalid schematic field parent item." ) );

    LIB_COMPONENT* entry = CMP_LIBRARY::FindLibraryComponent( component->GetLibName() );

    wxCHECK_RET( entry != NULL, wxT( "Library entry for component <" ) +
                 component->GetLibName() + wxT( "> could not be found." ) );

    fieldNdx = aField->GetId();

    if( fieldNdx == VALUE && entry->IsPower() )
    {
        DisplayInfoMessage( this, _( "The component is a POWER, it's value cannot be \
modified!\n\nYou must create a new power component with the value."  ) );
        return;
    }

    flag = 0;

    if( fieldNdx == REFERENCE && entry->GetPartCount() > 1 )
        flag = 1;

    /* save old cmp in undo list if not already in edit, or moving ... */
    if( aField->GetFlags() == 0 )
        SaveCopyInUndoList( component, UR_CHANGED );

    wxString newtext = aField->m_Text;
    DrawPanel->m_IgnoreMouseEvents = true;

    wxString title = _( "Field " ) + aField->m_Name;
    wxTextEntryDialog dlg( this, wxEmptyString , title, newtext );
    int diag = dlg.ShowModal();
    DrawPanel->MoveCursorToCrossHair();
    DrawPanel->m_IgnoreMouseEvents = false;
    newtext = dlg.GetValue( );
    newtext.Trim( true );
    newtext.Trim( false );

    if ( diag != wxID_OK || newtext == aField->GetText() )
        return;  // cancelled by user

    aField->m_AddExtraText = flag;
    aField->Draw( DrawPanel, aDC, wxPoint( 0, 0 ), g_XorMode );

    if( !newtext.IsEmpty() )
    {
        if( aField->m_Text.IsEmpty() )
        {
            aField->m_Pos = component->m_Pos;
            aField->m_Size.x = aField->m_Size.y = m_TextFieldSize;
        }

        aField->m_Text = newtext;

        if( fieldNdx == REFERENCE )
        {
            component->SetRef( GetSheet(), newtext );
        }
    }
    else
    {
        if( fieldNdx == REFERENCE )
        {
            DisplayError( this, _( "The reference field cannot be empty!  No change" ) );
        }
        else if( fieldNdx == VALUE )
        {
            DisplayError( this, _( "The value field cannot be empty!  No change" ) );
        }
        else
        {
            aField->m_Text = wxT( "~" );
        }
    }

    aField->Draw( DrawPanel, aDC, wxPoint( 0, 0 ), g_XorMode );
    component->DisplayInfo( this );
    OnModify();
}


/*
 * Move standard text field.  This routine is normally attached to the cursor.
 */
static void MoveCmpField( EDA_DRAW_PANEL* aPanel, wxDC* aDC, const wxPoint& aPosition,
                          bool aErase )
{
    wxPoint pos;
    int fieldNdx;

    SCH_EDIT_FRAME* frame = (SCH_EDIT_FRAME*) aPanel->GetParent();
    SCH_FIELD*      currentField = frame->GetCurrentField();

    if( currentField == NULL )
        return;

    SCH_COMPONENT* component = (SCH_COMPONENT*) currentField->GetParent();
    fieldNdx = currentField->m_FieldId;

    currentField->m_AddExtraText = frame->m_Multiflag;

    if( aErase )
    {
        currentField->Draw( aPanel, aDC, wxPoint( 0, 0 ), g_XorMode );
    }

    pos = ( (SCH_COMPONENT*) currentField->GetParent() )->m_Pos;

    // Actual positions are calculated by the rotation/mirror transform
    // But here we want the relative position of the moved field
    // and we know the actual position.
    // So we are using the inverse rotation/mirror transform.
    wxPoint pt( aPanel->GetScreen()->GetCrossHairPosition() - pos );

    TRANSFORM itrsfm = component->GetTransform().InverseTransform();
    currentField->m_Pos = pos + itrsfm.TransformCoordinate( pt );

    currentField->Draw( aPanel, aDC, wxPoint( 0, 0 ), g_XorMode );
}


static void AbortMoveCmpField( EDA_DRAW_PANEL* Panel, wxDC* DC )
{
    SCH_EDIT_FRAME* frame = (SCH_EDIT_FRAME*) Panel->GetParent();
    SCH_FIELD*      currentField = frame->GetCurrentField();

    if( currentField )
    {
        currentField->m_AddExtraText = frame->m_Multiflag;
        currentField->Draw( Panel, DC, wxPoint( 0, 0 ), g_XorMode );
        currentField->m_Flags = 0;
        currentField->m_Pos   = frame->m_OldPos;
        currentField->Draw( Panel, DC, wxPoint( 0, 0 ), g_XorMode );
    }

    frame->SetCurrentField( NULL );

    SAFE_DELETE( g_ItemToUndoCopy );
}


void SCH_EDIT_FRAME::RotateCmpField( SCH_FIELD* Field, wxDC* DC )
{
    int            fieldNdx, flag;
    LIB_COMPONENT* Entry;

    if( Field == NULL )
        return;
    if( Field->m_Text == wxEmptyString )
        return;

    SCH_COMPONENT* Cmp = (SCH_COMPONENT*) Field->GetParent();

    fieldNdx = Field->m_FieldId;
    flag     = 0;
    if( fieldNdx == REFERENCE )
    {
        Entry = CMP_LIBRARY::FindLibraryComponent(
            ( (SCH_COMPONENT*) Field->GetParent() )->GetLibName() );

        if( Entry != NULL )
        {
            if( Entry->GetPartCount() > 1 )
                flag = 1;
        }
    }

    /* save old cmp in undo list if not already in edit, or moving ... */
    if( Field->m_Flags == 0 )
        SaveCopyInUndoList( Cmp, UR_CHANGED );

    Field->m_AddExtraText = flag;
    Field->Draw( DrawPanel, DC, wxPoint( 0, 0 ), g_XorMode );

    if( Field->m_Orient == TEXT_ORIENT_HORIZ )
        Field->m_Orient = TEXT_ORIENT_VERT;
    else
        Field->m_Orient = TEXT_ORIENT_HORIZ;
    Field->Draw( DrawPanel, DC, wxPoint( 0, 0 ), g_XorMode );

    OnModify();
}


/****************************************************************************/
/* Edit the component text reference*/
/****************************************************************************/
void SCH_EDIT_FRAME::EditComponentReference( SCH_COMPONENT* Cmp, wxDC* DC )
{
    wxCHECK_RET( Cmp != NULL && Cmp->Type() == SCH_COMPONENT_T,
                 wxT( "Invalid schematic component item." ) );

    LIB_COMPONENT* Entry;
    int            flag = 0;

    Entry = CMP_LIBRARY::FindLibraryComponent( Cmp->GetLibName() );

    if( Entry == NULL )
        return;

    if( Entry->GetPartCount() > 1 )
        flag = 1;

    wxString ref = Cmp->GetRef( GetSheet() );
    wxTextEntryDialog dlg( this, _( "Reference" ), _( "Component reference" ), ref );
    if( dlg.ShowModal() != wxID_OK )
        return; // cancelled by user

    ref = dlg.GetValue( );
    ref.Trim( true );
    ref.Trim( false );

    if( !ref.IsEmpty() ) // New text entered
    {
        /* save old cmp in undo list if not already in edit, or moving ... */
        if( Cmp->m_Flags == 0 )
            SaveCopyInUndoList( Cmp, UR_CHANGED );
        Cmp->SetRef( GetSheet(), ref );

        Cmp->GetField( REFERENCE )->m_AddExtraText = flag;
        Cmp->GetField( REFERENCE )->Draw( DrawPanel, DC, wxPoint( 0, 0 ), g_XorMode );
        Cmp->SetRef( GetSheet(), ref );
        Cmp->GetField( REFERENCE )->Draw( DrawPanel, DC, wxPoint( 0, 0 ),
                                          Cmp->m_Flags ? g_XorMode : GR_DEFAULT_DRAWMODE );
        OnModify();
    }

    Cmp->DisplayInfo( this );
}


/*****************************************************************************/
/* Routine to change the selected text */
/*****************************************************************************/
void SCH_EDIT_FRAME::EditComponentValue( SCH_COMPONENT* Cmp, wxDC* DC )
{
    wxCHECK_RET( Cmp != NULL && Cmp->Type() == SCH_COMPONENT_T,
                 wxT( "Invalid schematic component item." ) );

    wxString       message;
    LIB_COMPONENT* Entry;

    Entry = CMP_LIBRARY::FindLibraryComponent( Cmp->GetLibName() );

    if( Entry == NULL )
        return;

    SCH_FIELD* TextField = Cmp->GetField( VALUE );

    message = TextField->m_Text;

    wxTextEntryDialog dlg( this,  _( "Value" ), _( "Component value" ), message );
    if( dlg.ShowModal() != wxID_OK )
        return; // cancelled by user

    message = dlg.GetValue( );
    message.Trim( true );
    message.Trim( false );

    if( !message.IsEmpty() )
    {
        /* save old cmp in undo list if not already in edit, or moving ... */
        if( Cmp->m_Flags == 0 )
            SaveCopyInUndoList( Cmp, UR_CHANGED );

        TextField->Draw( DrawPanel, DC, wxPoint( 0, 0 ), g_XorMode );
        TextField->m_Text = message;
        TextField->Draw( DrawPanel, DC, wxPoint( 0, 0 ),
                         Cmp->m_Flags ? g_XorMode : GR_DEFAULT_DRAWMODE );
        OnModify();
    }

    Cmp->DisplayInfo( this );
}


void SCH_EDIT_FRAME::EditComponentFootprint( SCH_COMPONENT* Cmp, wxDC* DC )
{
    wxCHECK_RET( Cmp != NULL && Cmp->Type() == SCH_COMPONENT_T,
                 wxT( "Invalid schematic component item." ) );

    wxString       message;
    LIB_COMPONENT* Entry;

    Entry = CMP_LIBRARY::FindLibraryComponent( Cmp->GetLibName() );

    if( Entry == NULL )
        return;

    SCH_FIELD* TextField = Cmp->GetField( FOOTPRINT );
    message = TextField->m_Text;

    wxTextEntryDialog dlg( this, _( "Footprint" ), _( "Component footprint" ), message );
    if( dlg.ShowModal() != wxID_OK )
        return; // cancelled by user

    message = dlg.GetValue( );
    message.Trim( true );
    message.Trim( false );

    bool wasEmpty = false;
    if( TextField->m_Text.IsEmpty() )
        wasEmpty = true;

    // save old cmp in undo list if not already in edit, or moving ...
    if( Cmp->m_Flags == 0 )
        SaveCopyInUndoList( Cmp, UR_CHANGED );
    Cmp->GetField( FOOTPRINT )->Draw( DrawPanel, DC, wxPoint( 0, 0 ), g_XorMode );

    // Give a suitable position to the field if it was new,
    // and therefore has no position already given.
    if( wasEmpty && !message.IsEmpty() )
    {
        Cmp->GetField( FOOTPRINT )->m_Pos = Cmp->GetField( REFERENCE )->m_Pos;

        // add offset here - ? suitable heuristic below?
        Cmp->GetField( FOOTPRINT )->m_Pos.x +=
            ( Cmp->GetField( REFERENCE )->m_Pos.x - Cmp->m_Pos.x ) > 0 ?
            ( Cmp->GetField( REFERENCE )->m_Size.x ) :
            ( -1 * Cmp->GetField( REFERENCE )->m_Size.x );

        Cmp->GetField( FOOTPRINT )->m_Pos.y +=
            ( Cmp->GetField( REFERENCE )->m_Pos.y - Cmp->m_Pos.y ) > 0 ?
            ( Cmp->GetField( REFERENCE )->m_Size.y ) :
            ( -1 * Cmp->GetField( REFERENCE )->m_Size.y );

        Cmp->GetField( FOOTPRINT )->m_Orient = Cmp->GetField( REFERENCE )->m_Orient;
    }

    TextField->m_Text = message;
    Cmp->GetField( FOOTPRINT )->Draw( DrawPanel, DC, wxPoint( 0, 0 ),
                                      Cmp->m_Flags ? g_XorMode : GR_DEFAULT_DRAWMODE );
    OnModify();

    Cmp->DisplayInfo( this );
}
