/****************************/
/* Trigonometric functions. */
/****************************/

#include "fctsys.h"
#include "trigo.h"

/**
 * Function TestSegmentHit
 * test for hit on line segment
 * i.e. cursor within a given distance from segment
 * @param aRefPoint = cursor (point to test) coords
 * @param aStart is the first end-point of the line segment
 * @param aEnd is the second end-point of the line segment
 * @param aDist = maximum distance for hit
 */
bool TestSegmentHit( wxPoint aRefPoint, wxPoint aStart, wxPoint aEnd, int aDist )
{
    // make coordinates relatives to aStart:
    aEnd -= aStart;
    aRefPoint -= aStart;
    return DistanceTest( aDist, aEnd.x, aEnd.y, aRefPoint.x, aRefPoint.y );
}


/**
 * Function DistanceTest
 * Calculate the distance from mouse cursor to a line segment.
 * Returns:
 * False if distance > threshold
 * True if distance <= threshold
 */
bool DistanceTest( int seuil, int dx, int dy, int spot_cX, int spot_cY )
{
/*  We can have 4 cases::
 *      horizontal segment
 *      vertical segment
 *      45 degrees segment
 *      other slopes
 */
    int cXrot, cYrot, segX, segY;
    int pointX, pointY;

    segX   = dx;
    segY   = dy;
    pointX = spot_cX;
    pointY = spot_cY;

    /*Recalculating coord for the segment is in 1st quadrant (coord >= 0)*/
    if( segX < 0 )   /* set > 0 by symmetry about the Y axis */
    {
        segX   = -segX;
        pointX = -pointX;
    }
    if( segY < 0 )   /* set > 0 by symmetry about the X axis */
    {
        segY   = -segY;
        pointY = -pointY;
    }


    if( segY == 0 ) /* horizontal */
    {
        if( abs( pointY ) <= seuil )
        {
            if( ( pointX >= 0 ) && ( pointX <= segX ) )
                return 1;

            if( ( pointX < 0 ) && ( pointX >= -seuil ) )
            {
                if( ( ( pointX * pointX ) + ( pointY * pointY ) )
                   <= ( seuil * seuil ) )
                    return true;
            }
            if( ( pointX > segX ) && ( pointX <= ( segX + seuil ) ) )
            {
                if( ( ( ( pointX - segX ) * ( pointX - segX ) )
                     + ( pointY * pointY ) ) <= ( seuil * seuil ) )
                    return true;
            }
        }
    }
    else if( segX == 0 ) /* vertical */
    {
        if( abs( pointX ) <= seuil )
        {
            if( ( pointY >= 0 ) && ( pointY <= segY ) )
                return true;
            if( ( pointY < 0 ) && ( pointY >= -seuil ) )
            {
                if( ( ( pointY * pointY ) + ( pointX * pointX ) )
                   <= ( seuil * seuil ) )
                    return true;
            }
            if( ( pointY > segY ) && ( pointY <= ( segY + seuil ) ) )
            {
                if( ( ( ( pointY - segY ) * ( pointY - segY ) )
                     + ( pointX * pointX ) ) <= ( seuil * seuil ) )
                    return true;
            }
        }
    }
    else if( segX == segY )    /* 45 degrees */
    {
        /* Rotate axes of 45 degrees. mouse was then
         * Coord: x1 = x * y * cos45 + sin45
         * y1 = y * cos45 - sin45 x *
         * And the segment of track is horizontal.
         * Coord recalculation of the mouse (sin45 = cos45 = .707 = 7 / 10
         * Note: sin or cos45 = .707, and when recalculating coord
         * dx45 and dy45, lect coeff .707 is neglected, dx and dy are
         * actually 0707 times
         * Too big. (security hole too small)
         * Spot_cX, Y * must be by .707 * .707 = 0.5
         */

        cXrot = (pointX + pointY) >> 1;
        cYrot = (pointY - pointX) >> 1;

        /* Recalculating coord of segment extremity, which will be vertical
         * following the orientation of axes on the screen: dx45 = pointx
         * (or pointy) and 1.414 is actually greater, and dy45 = 0
         */

        // * Threshold should be .707 to reflect the change in coeff dx, dy
        seuil *= 7;
        seuil /= 10;

        if( abs( cYrot ) <= seuil ) /* ok on vertical axis */
        {
            if( ( cXrot >= 0 ) && ( cXrot <= segX ) )
                return true;

            /* Check extremes using the radius of a circle. */
            if( ( cXrot < 0 ) && ( cXrot >= -seuil ) )
            {
                if( ( ( cXrot * cXrot ) + ( cYrot * cYrot ) )
                   <= ( seuil * seuil ) )
                    return true;
            }
            if( ( cXrot > segX ) && ( cXrot <= ( segX + seuil ) ) )
            {
                if( ( ( ( cXrot - segX ) * ( cXrot - segX ) )
                     + ( cYrot * cYrot ) ) <= ( seuil * seuil ) )
                    return true;
            }
        }
    }
    else    /* any orientation */
    {
        /* There is a change of axis (rotation), so that the segment
         * track is horizontal in the new reference
         */
        int angle;

        angle = wxRound( ( atan2( (double) segY,
                                 (double) segX ) * 1800.0 / M_PI ) );
        cXrot = pointX;
        cYrot = pointY;

        RotatePoint( &cXrot, &cYrot, angle );   /* Rotate the point to be tested */
        RotatePoint( &segX, &segY, angle );     /* Rotate the segment */

        /* The track is horizontal, following the amendments to coordinate
         * axis and, therefore segX = length of segment
         */
        if( abs( cYrot ) <= seuil ) /* vertical axis */
        {
            if( ( cXrot >= 0 ) && ( cXrot <= segX ) )
                return true;

            if( ( cXrot < 0 ) && ( cXrot >= -seuil ) )
            {
                if( ( ( cXrot * cXrot ) + ( cYrot * cYrot ) )
                   <= ( seuil * seuil ) )
                    return true;
            }
            if( ( cXrot > segX ) && ( cXrot <= ( segX + seuil ) ) )
            {
                if( ( ( ( cXrot - segX ) * ( cXrot - segX ) )
                     + ( cYrot * cYrot ) ) <= ( seuil * seuil ) )
                    return true;
            }
        }
    }

    return false;
}


/* Return the arc tangent of 0.1 degrees coord vector dx, dy
 * between -1800 and 1800
 * Equivalent to atan2 (but faster for calculations if
 * the angle is 0 to -1800, or + - 900
 */
int ArcTangente( int dy, int dx )
{
    double fangle;

    if( dy == 0 )
    {
        if( dx >= 0 )
            return 0;
        else
            return -1800;
    }

    if( dx == 0 )
    {
        if( dy >= 0 )
            return 900;
        else
            return -900;
    }

    if( dx == dy )
    {
        if( dx >= 0 )
            return 450;
        else
            return -1800 + 450;
    }

    if( dx == -dy )
    {
        if( dx >= 0 )
            return -450;
        else
            return 1800 - 450;
    }

    fangle = atan2( (double) dy, (double) dx ) / M_PI * 1800;
    return wxRound( fangle );
}


/*
 * Calculate the new point of coord coord pX, pY,
 * for a rotation center 0, 0, and angle in (1 / 10 degree)
 */
void RotatePoint( int* pX, int* pY, int angle )
{
    double fpx, fpy;
    int    tmp;

    while( angle < 0 )
        angle += 3600;

    while( angle >= 3600 )
        angle -= 3600;

    if( angle == 0 )
        return;

    /*  coord:  xrot = y*sin + x*cos
     *          yrot = y*cos - x*sin
     */
    if( angle == 900 )          /* sin = 1, cos = 0 */
    {
        tmp = *pX;
        *pX = *pY;
        *pY = -tmp;
    }
    else if( angle == 1800 )    /* sin = 0, cos = -1 */
    {
        *pX = -*pX;
        *pY = -*pY;
    }
    else if( angle == 2700 )    /* sin = -1, cos = 0 */
    {
        tmp = *pX;
        *pX = -*pY;
        *pY = tmp;
    }
    else
    {
        fpx = ( *pY * fsinus[angle] ) + ( *pX * fcosinus[angle] );
        fpy = ( *pY * fcosinus[angle] ) - ( *pX * fsinus[angle] );

        *pX = wxRound( fpx );
        *pY = wxRound( fpy );
    }
}


/*
 * Calculate the new point of coord coord pX, pY,
 * for a rotation center cx, cy, and angle in (1 / 10 degree)
 */
void RotatePoint( int* pX, int* pY, int cx, int cy, int angle )
{
    int ox, oy;

    ox = *pX - cx;
    oy = *pY - cy;

    RotatePoint( &ox, &oy, angle );

    *pX = ox + cx;
    *pY = oy + cy;
}


/*
 * Calculates the new coord point point
 * for a rotation angle in (1 / 10 degree)
 */
void RotatePoint( wxPoint* point, int angle )
{
    int ox, oy;

    ox = point->x;
    oy = point->y;

    RotatePoint( &ox, &oy, angle );
    point->x = ox;
    point->y = oy;
}


/*
 * Calculates the new coord point point
 * for a center rotation center and angle in (1 / 10 degree)
 */
void RotatePoint( wxPoint* point, const wxPoint& centre, int angle )
{
    int ox, oy;

    ox = point->x - centre.x;
    oy = point->y - centre.y;

    RotatePoint( &ox, &oy, angle );
    point->x = ox + centre.x;
    point->y = oy + centre.y;
}


void RotatePoint( double* pX, double* pY, double cx, double cy, int angle )
{
    double ox, oy;

    ox = *pX - cx;
    oy = *pY - cy;

    RotatePoint( &ox, &oy, angle );

    *pX = ox + cx;
    *pY = oy + cy;
}


void RotatePoint( double* pX, double* pY, int angle )
{
    double tmp;

    while( angle < 0 )
        angle += 3600;

    while( angle >= 3600 )
        angle -= 3600;

    if( angle == 0 )
        return;

    if( angle == 900 )          /* sin = 1, cos = 0 */
    {
        tmp = *pX;
        *pX = *pY;
        *pY = -tmp;
    }
    else if( angle == 1800 )    /* sin = 0, cos = -1 */
    {
        *pX = -*pX;
        *pY = -*pY;
    }
    else if( angle == 2700 )    /* sin = -1, cos = 0 */
    {
        tmp = *pX;
        *pX = -*pY;
        *pY = tmp;
    }
    else
    {
        double fpx = (*pY * fsinus[angle]) + (*pX * fcosinus[angle]);
        double fpy = (*pY * fcosinus[angle]) - (*pX * fsinus[angle]);

        *pX = fpx;
        *pY = fpy;
    }
}


double EuclideanNorm( wxPoint vector )
{
    return sqrt( (double) vector.x * (double) vector.x + (double) vector.y * (double) vector.y );
}


wxPoint TwoPointVector( wxPoint startPoint, wxPoint endPoint )
{
    return endPoint - startPoint;
}


double DistanceLinePoint( wxPoint linePointA, wxPoint linePointB, wxPoint referencePoint )
{
    return fabs( (double) ( (linePointB.x - linePointA.x) * (linePointA.y - referencePoint.y) -
                 (linePointA.x - referencePoint.x ) * (linePointB.y - linePointA.y) )
                  / EuclideanNorm( TwoPointVector( linePointA, linePointB ) ) );
}


bool HitTestPoints( wxPoint pointA, wxPoint pointB, double threshold )
{
    wxPoint vectorAB = TwoPointVector( pointA, pointB );
    double  distance = EuclideanNorm( vectorAB );

    return distance < threshold;
}


int CrossProduct( wxPoint vectorA, wxPoint vectorB )
{
    return vectorA.x * vectorB.y - vectorA.y * vectorB.x;
}


double GetLineLength( const wxPoint& aPointA, const wxPoint& aPointB )
{
    return sqrt( pow( (double) aPointA.x - (double) aPointB.x, 2 ) +
                 pow( (double) aPointA.y - (double) aPointB.y, 2 ) );
}


double fsinus[3600] =
{
    0.0000000000,  0.0017453284,  0.0034906514,  0.0052359638,
    0.0069812603,  0.0087265355,  0.0104717841,  0.0122170008, 0.0139621803,
    0.0157073173,  0.0174524064,  0.0191974424,  0.0209424199, 0.0226873336,
    0.0244321782,  0.0261769483,  0.0279216387,  0.0296662441, 0.0314107591,
    0.0331551784,  0.0348994967,  0.0366437087,  0.0383878091, 0.0401317925,
    0.0418756537,  0.0436193874,  0.0453629881,  0.0471064507, 0.0488497698,
    0.0505929401,  0.0523359562,  0.0540788130,  0.0558215050, 0.0575640270,
    0.0593063736,  0.0610485395,  0.0627905195,  0.0645323083, 0.0662739004,
    0.0680152907,  0.0697564737,  0.0714974443,  0.0732381971, 0.0749787268,
    0.0767190281,  0.0784590957,  0.0801989243,  0.0819385086, 0.0836778433,
    0.0854169231,  0.0871557427,  0.0888942969,  0.0906325802, 0.0923705874,
    0.0941083133,  0.0958457525,  0.0975828998,  0.0993197497, 0.1010562972,
    0.1027925368,  0.1045284633,  0.1062640713,  0.1079993557, 0.1097343111,
    0.1114689322,  0.1132032138,  0.1149371505,  0.1166707371, 0.1184039683,
    0.1201368388,  0.1218693434,  0.1236014767,  0.1253332336, 0.1270646086,
    0.1287955966,  0.1305261922,  0.1322563903,  0.1339861854, 0.1357155724,
    0.1374445460,  0.1391731010,  0.1409012319,  0.1426289337, 0.1443562010,
    0.1460830286,  0.1478094111,  0.1495353434,  0.1512608202, 0.1529858363,
    0.1547103863,  0.1564344650,  0.1581580673,  0.1598811877, 0.1616038211,
    0.1633259622,  0.1650476059,  0.1667687467,  0.1684893796, 0.1702094992,
    0.1719291003,  0.1736481777,  0.1753667261,  0.1770847403, 0.1788022151,
    0.1805191453,  0.1822355255,  0.1839513506,  0.1856666154, 0.1873813146,
    0.1890954430,  0.1908089954,  0.1925219665,  0.1942343512, 0.1959461442,
    0.1976573404,  0.1993679344,  0.2010779211,  0.2027872954, 0.2044960518,
    0.2062041854,  0.2079116908,  0.2096185629,  0.2113247965, 0.2130303863,
    0.2147353272,  0.2164396139,  0.2181432414,  0.2198462044, 0.2215484976,
    0.2232501160,  0.2249510543,  0.2266513074,  0.2283508701, 0.2300497372,
    0.2317479035,  0.2334453639,  0.2351421131,  0.2368381461, 0.2385334576,
    0.2402280425,  0.2419218956,  0.2436150118,  0.2453073859, 0.2469990127,
    0.2486898872,  0.2503800041,  0.2520693582,  0.2537579446, 0.2554457579,
    0.2571327932,  0.2588190451,  0.2605045086,  0.2621891786, 0.2638730500,
    0.2655561175,  0.2672383761,  0.2689198206,  0.2706004460, 0.2722802470,
    0.2739592187,  0.2756373558,  0.2773146533,  0.2789911060, 0.2806667089,
    0.2823414568,  0.2840153447,  0.2856883674,  0.2873605198, 0.2890317969,
    0.2907021936,  0.2923717047,  0.2940403252,  0.2957080500, 0.2973748741,
    0.2990407923,  0.3007057995,  0.3023698908,  0.3040330609, 0.3056953050,
    0.3073566178,  0.3090169944,  0.3106764296,  0.3123349185, 0.3139924560,
    0.3156490369,  0.3173046564,  0.3189593093,  0.3206129906, 0.3222656952,
    0.3239174182,  0.3255681545,  0.3272178990,  0.3288666467, 0.3305143927,
    0.3321611319,  0.3338068592,  0.3354515698,  0.3370952584, 0.3387379202,
    0.3403795502,  0.3420201433,  0.3436596946,  0.3452981990, 0.3469356516,
    0.3485720473,  0.3502073813,  0.3518416484,  0.3534748438, 0.3551069624,
    0.3567379993,  0.3583679495,  0.3599968081,  0.3616245701, 0.3632512305,
    0.3648767843,  0.3665012267,  0.3681245527,  0.3697467573, 0.3713678356,
    0.3729877826,  0.3746065934,  0.3762242631,  0.3778407868, 0.3794561595,
    0.3810703764,  0.3826834324,  0.3842953227,  0.3859060423, 0.3875155865,
    0.3891239501,  0.3907311285,  0.3923371166,  0.3939419096, 0.3955455026,
    0.3971478906,  0.3987490689,  0.4003490326,  0.4019477767, 0.4035452964,
    0.4051415868,  0.4067366431,  0.4083304604,  0.4099230338, 0.4115143586,
    0.4131044298,  0.4146932427,  0.4162807923,  0.4178670738, 0.4194520824,
    0.4210358134,  0.4226182617,  0.4241994227,  0.4257792916, 0.4273578634,
    0.4289351334,  0.4305110968,  0.4320857488,  0.4336590846, 0.4352310994,
    0.4368017884,  0.4383711468,  0.4399391699,  0.4415058528, 0.4430711908,
    0.4446351792,  0.4461978131,  0.4477590878,  0.4493189986, 0.4508775407,
    0.4524347093,  0.4539904997,  0.4555449072,  0.4570979271, 0.4586495545,
    0.4601997848,  0.4617486132,  0.4632960351,  0.4648420457, 0.4663866403,
    0.4679298143,  0.4694715628,  0.4710118812,  0.4725507649, 0.4740882090,
    0.4756242091,  0.4771587603,  0.4786918579,  0.4802234974, 0.4817536741,
    0.4832823833,  0.4848096202,  0.4863353804,  0.4878596591, 0.4893824517,
    0.4909037536,  0.4924235601,  0.4939418666,  0.4954586684, 0.4969739610,
    0.4984877398,  0.5000000000,  0.5015107372,  0.5030199466, 0.5045276238,
    0.5060337641,  0.5075383630,  0.5090414158,  0.5105429179, 0.5120428649,
    0.5135412521,  0.5150380749,  0.5165333289,  0.5180270094, 0.5195191119,
    0.5210096318,  0.5224985647,  0.5239859060,  0.5254716511, 0.5269557955,
    0.5284383347,  0.5299192642,  0.5313985795,  0.5328762761, 0.5343523494,
    0.5358267950,  0.5372996083,  0.5387707850,  0.5402403205, 0.5417082103,
    0.5431744500,  0.5446390350,  0.5461019610,  0.5475632235, 0.5490228180,
    0.5504807401,  0.5519369853,  0.5533915492,  0.5548444274, 0.5562956155,
    0.5577451090,  0.5591929035,  0.5606389946,  0.5620833779, 0.5635260489,
    0.5649670034,  0.5664062369,  0.5678437451,  0.5692795234, 0.5707135677,
    0.5721458734,  0.5735764364,  0.5750052520,  0.5764323162, 0.5778576244,
    0.5792811723,  0.5807029557,  0.5821229702,  0.5835412114, 0.5849576750,
    0.5863723567,  0.5877852523,  0.5891963574,  0.5906056676, 0.5920131788,
    0.5934188866,  0.5948227868,  0.5962248750,  0.5976251470, 0.5990235985,
    0.6004202253,  0.6018150232,  0.6032079877,  0.6045991149, 0.6059884003,
    0.6073758397,  0.6087614290,  0.6101451639,  0.6115270402, 0.6129070537,
    0.6142852001,  0.6156614753,  0.6170358751,  0.6184083954, 0.6197790318,
    0.6211477803,  0.6225146366,  0.6238795967,  0.6252426563, 0.6266038114,
    0.6279630576,  0.6293203910,  0.6306758074,  0.6320293027, 0.6333808726,
    0.6347305132,  0.6360782203,  0.6374239897,  0.6387678175, 0.6401096995,
    0.6414496316,  0.6427876097,  0.6441236298,  0.6454576877, 0.6467897795,
    0.6481199011,  0.6494480483,  0.6507742173,  0.6520984038, 0.6534206040,
    0.6547408137,  0.6560590290,  0.6573752458,  0.6586894601, 0.6600016680,
    0.6613118653,  0.6626200482,  0.6639262127,  0.6652303547, 0.6665324702,
    0.6678325555,  0.6691306064,  0.6704266190,  0.6717205893, 0.6730125135,
    0.6743023876,  0.6755902076,  0.6768759697,  0.6781596699, 0.6794413043,
    0.6807208690,  0.6819983601,  0.6832737737,  0.6845471059, 0.6858183529,
    0.6870875108,  0.6883545757,  0.6896195437,  0.6908824111, 0.6921431739,
    0.6934018283,  0.6946583705,  0.6959127966,  0.6971651029, 0.6984152854,
    0.6996633405,  0.7009092643,  0.7021530530,  0.7033947028, 0.7046342100,
    0.7058715707,  0.7071067812,  0.7083398377,  0.7095707365, 0.7107994739,
    0.7120260460,  0.7132504492,  0.7144726796,  0.7156927337, 0.7169106077,
    0.7181262978,  0.7193398003,  0.7205511117,  0.7217602281, 0.7229671459,
    0.7241718614,  0.7253743710,  0.7265746710,  0.7277727577, 0.7289686274,
    0.7301622766,  0.7313537016,  0.7325428988,  0.7337298645, 0.7349145951,
    0.7360970871,  0.7372773368,  0.7384553406,  0.7396310950, 0.7408045963,
    0.7419758410,  0.7431448255,  0.7443115462,  0.7454759997, 0.7466381823,
    0.7477980905,  0.7489557208,  0.7501110696,  0.7512641335, 0.7524149089,
    0.7535633923,  0.7547095802,  0.7558534692,  0.7569950557, 0.7581343362,
    0.7592713073,  0.7604059656,  0.7615383075,  0.7626683297, 0.7637960286,
    0.7649214009,  0.7660444431,  0.7671651518,  0.7682835236, 0.7693995550,
    0.7705132428,  0.7716245834,  0.7727335735,  0.7738402097, 0.7749444887,
    0.7760464071,  0.7771459615,  0.7782431485,  0.7793379649, 0.7804304073,
    0.7815204724,  0.7826081569,  0.7836934573,  0.7847763705, 0.7858568932,
    0.7869350220,  0.7880107536,  0.7890840848,  0.7901550124, 0.7912235330,
    0.7922896434,  0.7933533403,  0.7944146205,  0.7954734809, 0.7965299180,
    0.7975839288,  0.7986355100,  0.7996846585,  0.8007313709, 0.8017756442,
    0.8028174752,  0.8038568606,  0.8048937974,  0.8059282822, 0.8069603121,
    0.8079898839,  0.8090169944,  0.8100416404,  0.8110638190, 0.8120835269,
    0.8131007610,  0.8141155184,  0.8151277957,  0.8161375901, 0.8171448983,
    0.8181497174,  0.8191520443,  0.8201518759,  0.8211492091, 0.8221440410,
    0.8231363685,  0.8241261886,  0.8251134983,  0.8260982945, 0.8270805743,
    0.8280603346,  0.8290375726,  0.8300122851,  0.8309844693, 0.8319541221,
    0.8329212407,  0.8338858221,  0.8348478633,  0.8358073614, 0.8367643135,
    0.8377187166,  0.8386705679,  0.8396198645,  0.8405666035, 0.8415107819,
    0.8424523970,  0.8433914458,  0.8443279255,  0.8452618332, 0.8461931661,
    0.8471219214,  0.8480480962,  0.8489716876,  0.8498926930, 0.8508111094,
    0.8517269341,  0.8526401644,  0.8535507973,  0.8544588301, 0.8553642602,
    0.8562670846,  0.8571673007,  0.8580649057,  0.8589598969, 0.8598522716,
    0.8607420270,  0.8616291604,  0.8625136692,  0.8633955506, 0.8642748020,
    0.8651514206,  0.8660254038,  0.8668967489,  0.8677654534, 0.8686315144,
    0.8694949295,  0.8703556959,  0.8712138111,  0.8720692724, 0.8729220773,
    0.8737722230,  0.8746197071,  0.8754645270,  0.8763066800, 0.8771461637,
    0.8779829754,  0.8788171127,  0.8796485729,  0.8804773535, 0.8813034521,
    0.8821268660,  0.8829475929,  0.8837656301,  0.8845809752, 0.8853936258,
    0.8862035792,  0.8870108332,  0.8878153851,  0.8886172327, 0.8894163733,
    0.8902128046,  0.8910065242,  0.8917975296,  0.8925858185, 0.8933713883,
    0.8941542368,  0.8949343616,  0.8957117602,  0.8964864304, 0.8972583697,
    0.8980275758,  0.8987940463,  0.8995577790,  0.9003187714, 0.9010770213,
    0.9018325264,  0.9025852843,  0.9033352929,  0.9040825497, 0.9048270525,
    0.9055687990,  0.9063077870,  0.9070440143,  0.9077774785, 0.9085081775,
    0.9092361090,  0.9099612709,  0.9106836608,  0.9114032766, 0.9121201162,
    0.9128341772,  0.9135454576,  0.9142539552,  0.9149596678, 0.9156625933,
    0.9163627296,  0.9170600744,  0.9177546257,  0.9184463813, 0.9191353393,
    0.9198214973,  0.9205048535,  0.9211854056,  0.9218631516, 0.9225380895,
    0.9232102171,  0.9238795325,  0.9245460336,  0.9252097184, 0.9258705848,
    0.9265286309,  0.9271838546,  0.9278362539,  0.9284858269, 0.9291325715,
    0.9297764859,  0.9304175680,  0.9310558159,  0.9316912276, 0.9323238012,
    0.9329535348,  0.9335804265,  0.9342044743,  0.9348256764, 0.9354440308,
    0.9360595357,  0.9366721892,  0.9372819895,  0.9378889346, 0.9384930228,
    0.9390942521,  0.9396926208,  0.9402881270,  0.9408807690, 0.9414705448,
    0.9420574528,  0.9426414911,  0.9432226579,  0.9438009516, 0.9443763702,
    0.9449489122,  0.9455185756,  0.9460853588,  0.9466492601, 0.9472102777,
    0.9477684100,  0.9483236552,  0.9488760116,  0.9494254776, 0.9499720515,
    0.9505157316,  0.9510565163,  0.9515944039,  0.9521293927, 0.9526614813,
    0.9531906678,  0.9537169507,  0.9542403285,  0.9547607995, 0.9552783621,
    0.9557930148,  0.9563047560,  0.9568135841,  0.9573194975, 0.9578224948,
    0.9583225745,  0.9588197349,  0.9593139745,  0.9598052920, 0.9602936857,
    0.9607791542,  0.9612616959,  0.9617413095,  0.9622179935, 0.9626917464,
    0.9631625668,  0.9636304532,  0.9640954042,  0.9645574185, 0.9650164945,
    0.9654726309,  0.9659258263,  0.9663760793,  0.9668233886, 0.9672677528,
    0.9677091705,  0.9681476404,  0.9685831611,  0.9690157314, 0.9694453499,
    0.9698720153,  0.9702957263,  0.9707164816,  0.9711342799, 0.9715491200,
    0.9719610006,  0.9723699204,  0.9727758782,  0.9731788728, 0.9735789029,
    0.9739759673,  0.9743700648,  0.9747611942,  0.9751493543, 0.9755345439,
    0.9759167619,  0.9762960071,  0.9766722783,  0.9770455744, 0.9774158943,
    0.9777832368,  0.9781476007,  0.9785089851,  0.9788673888, 0.9792228106,
    0.9795752496,  0.9799247046,  0.9802711746,  0.9806146585, 0.9809551553,
    0.9812926640,  0.9816271834,  0.9819587127,  0.9822872507, 0.9826127965,
    0.9829353491,  0.9832549076,  0.9835714708,  0.9838850379, 0.9841956080,
    0.9845031800,  0.9848077530,  0.9851093262,  0.9854078985, 0.9857034691,
    0.9859960371,  0.9862856015,  0.9865721616,  0.9868557164, 0.9871362651,
    0.9874138068,  0.9876883406,  0.9879598658,  0.9882283814, 0.9884938868,
    0.9887563810,  0.9890158634,  0.9892723330,  0.9895257891, 0.9897762309,
    0.9900236577,  0.9902680687,  0.9905094632,  0.9907478405, 0.9909831997,
    0.9912155403,  0.9914448614,  0.9916711624,  0.9918944426, 0.9921147013,
    0.9923319379,  0.9925461516,  0.9927573419,  0.9929655081, 0.9931706495,
    0.9933727656,  0.9935718557,  0.9937679192,  0.9939609555, 0.9941509640,
    0.9943379441,  0.9945218954,  0.9947028171,  0.9948807088, 0.9950555700,
    0.9952274000,  0.9953961984,  0.9955619646,  0.9957246982, 0.9958843986,
    0.9960410654,  0.9961946981,  0.9963452962,  0.9964928592, 0.9966373868,
    0.9967788785,  0.9969173337,  0.9970527522,  0.9971851335, 0.9973144772,
    0.9974407829,  0.9975640503,  0.9976842788,  0.9978014683, 0.9979156183,
    0.9980267284,  0.9981347984,  0.9982398279,  0.9983418166, 0.9984407642,
    0.9985366703,  0.9986295348,  0.9987193572,  0.9988061373, 0.9988898750,
    0.9989705698,  0.9990482216,  0.9991228301,  0.9991943951, 0.9992629164,
    0.9993283938,  0.9993908270,  0.9994502159,  0.9995065604, 0.9995598601,
    0.9996101150,  0.9996573250,  0.9997014898,  0.9997426093, 0.9997806835,
    0.9998157121,  0.9998476952,  0.9998766325,  0.9999025240, 0.9999253697,
    0.9999451694,  0.9999619231,  0.9999756307,  0.9999862922, 0.9999939077,
    0.9999984769,  1.0000000000,  0.9999984769,  0.9999939077, 0.9999862922,
    0.9999756307,  0.9999619231,  0.9999451694,  0.9999253697, 0.9999025240,
    0.9998766325,  0.9998476952,  0.9998157121,  0.9997806835, 0.9997426093,
    0.9997014898,  0.9996573250,  0.9996101150,  0.9995598601, 0.9995065604,
    0.9994502159,  0.9993908270,  0.9993283938,  0.9992629164, 0.9991943951,
    0.9991228301,  0.9990482216,  0.9989705698,  0.9988898750, 0.9988061373,
    0.9987193572,  0.9986295348,  0.9985366703,  0.9984407642, 0.9983418166,
    0.9982398279,  0.9981347984,  0.9980267284,  0.9979156183, 0.9978014683,
    0.9976842788,  0.9975640503,  0.9974407829,  0.9973144772, 0.9971851335,
    0.9970527522,  0.9969173337,  0.9967788785,  0.9966373868, 0.9964928592,
    0.9963452962,  0.9961946981,  0.9960410654,  0.9958843986, 0.9957246982,
    0.9955619646,  0.9953961984,  0.9952274000,  0.9950555700, 0.9948807088,
    0.9947028171,  0.9945218954,  0.9943379441,  0.9941509640, 0.9939609555,
    0.9937679192,  0.9935718557,  0.9933727656,  0.9931706495, 0.9929655081,
    0.9927573419,  0.9925461516,  0.9923319379,  0.9921147013, 0.9918944426,
    0.9916711624,  0.9914448614,  0.9912155403,  0.9909831997, 0.9907478405,
    0.9905094632,  0.9902680687,  0.9900236577,  0.9897762309, 0.9895257891,
    0.9892723330,  0.9890158634,  0.9887563810,  0.9884938868, 0.9882283814,
    0.9879598658,  0.9876883406,  0.9874138068,  0.9871362651, 0.9868557164,
    0.9865721616,  0.9862856015,  0.9859960371,  0.9857034691, 0.9854078985,
    0.9851093262,  0.9848077530,  0.9845031800,  0.9841956080, 0.9838850379,
    0.9835714708,  0.9832549076,  0.9829353491,  0.9826127965, 0.9822872507,
    0.9819587127,  0.9816271834,  0.9812926640,  0.9809551553, 0.9806146585,
    0.9802711746,  0.9799247046,  0.9795752496,  0.9792228106, 0.9788673888,
    0.9785089851,  0.9781476007,  0.9777832368,  0.9774158943, 0.9770455744,
    0.9766722783,  0.9762960071,  0.9759167619,  0.9755345439, 0.9751493543,
    0.9747611942,  0.9743700648,  0.9739759673,  0.9735789029, 0.9731788728,
    0.9727758782,  0.9723699204,  0.9719610006,  0.9715491200, 0.9711342799,
    0.9707164816,  0.9702957263,  0.9698720153,  0.9694453499, 0.9690157314,
    0.9685831611,  0.9681476404,  0.9677091705,  0.9672677528, 0.9668233886,
    0.9663760793,  0.9659258263,  0.9654726309,  0.9650164945, 0.9645574185,
    0.9640954042,  0.9636304532,  0.9631625668,  0.9626917464, 0.9622179935,
    0.9617413095,  0.9612616959,  0.9607791542,  0.9602936857, 0.9598052920,
    0.9593139745,  0.9588197349,  0.9583225745,  0.9578224948, 0.9573194975,
    0.9568135841,  0.9563047560,  0.9557930148,  0.9552783621, 0.9547607995,
    0.9542403285,  0.9537169507,  0.9531906678,  0.9526614813, 0.9521293927,
    0.9515944039,  0.9510565163,  0.9505157316,  0.9499720515, 0.9494254776,
    0.9488760116,  0.9483236552,  0.9477684100,  0.9472102777, 0.9466492601,
    0.9460853588,  0.9455185756,  0.9449489122,  0.9443763702, 0.9438009516,
    0.9432226579,  0.9426414911,  0.9420574528,  0.9414705448, 0.9408807690,
    0.9402881270,  0.9396926208,  0.9390942521,  0.9384930228, 0.9378889346,
    0.9372819895,  0.9366721892,  0.9360595357,  0.9354440308, 0.9348256764,
    0.9342044743,  0.9335804265,  0.9329535348,  0.9323238012, 0.9316912276,
    0.9310558159,  0.9304175680,  0.9297764859,  0.9291325715, 0.9284858269,
    0.9278362539,  0.9271838546,  0.9265286309,  0.9258705848, 0.9252097184,
    0.9245460336,  0.9238795325,  0.9232102171,  0.9225380895, 0.9218631516,
    0.9211854056,  0.9205048535,  0.9198214973,  0.9191353393, 0.9184463813,
    0.9177546257,  0.9170600744,  0.9163627296,  0.9156625933, 0.9149596678,
    0.9142539552,  0.9135454576,  0.9128341772,  0.9121201162, 0.9114032766,
    0.9106836608,  0.9099612709,  0.9092361090,  0.9085081775, 0.9077774785,
    0.9070440143,  0.9063077870,  0.9055687990,  0.9048270525, 0.9040825497,
    0.9033352929,  0.9025852843,  0.9018325264,  0.9010770213, 0.9003187714,
    0.8995577790,  0.8987940463,  0.8980275758,  0.8972583697, 0.8964864304,
    0.8957117602,  0.8949343616,  0.8941542368,  0.8933713883, 0.8925858185,
    0.8917975296,  0.8910065242,  0.8902128046,  0.8894163733, 0.8886172327,
    0.8878153851,  0.8870108332,  0.8862035792,  0.8853936258, 0.8845809752,
    0.8837656301,  0.8829475929,  0.8821268660,  0.8813034521, 0.8804773535,
    0.8796485729,  0.8788171127,  0.8779829754,  0.8771461637, 0.8763066800,
    0.8754645270,  0.8746197071,  0.8737722230,  0.8729220773, 0.8720692724,
    0.8712138111,  0.8703556959,  0.8694949295,  0.8686315144, 0.8677654534,
    0.8668967489,  0.8660254038,  0.8651514206,  0.8642748020, 0.8633955506,
    0.8625136692,  0.8616291604,  0.8607420270,  0.8598522716, 0.8589598969,
    0.8580649057,  0.8571673007,  0.8562670846,  0.8553642602, 0.8544588301,
    0.8535507973,  0.8526401644,  0.8517269341,  0.8508111094, 0.8498926930,
    0.8489716876,  0.8480480962,  0.8471219214,  0.8461931661, 0.8452618332,
    0.8443279255,  0.8433914458,  0.8424523970,  0.8415107819, 0.8405666035,
    0.8396198645,  0.8386705679,  0.8377187166,  0.8367643135, 0.8358073614,
    0.8348478633,  0.8338858221,  0.8329212407,  0.8319541221, 0.8309844693,
    0.8300122851,  0.8290375726,  0.8280603346,  0.8270805743, 0.8260982945,
    0.8251134983,  0.8241261886,  0.8231363685,  0.8221440410, 0.8211492091,
    0.8201518759,  0.8191520443,  0.8181497174,  0.8171448983, 0.8161375901,
    0.8151277957,  0.8141155184,  0.8131007610,  0.8120835269, 0.8110638190,
    0.8100416404,  0.8090169944,  0.8079898839,  0.8069603121, 0.8059282822,
    0.8048937974,  0.8038568606,  0.8028174752,  0.8017756442, 0.8007313709,
    0.7996846585,  0.7986355100,  0.7975839288,  0.7965299180, 0.7954734809,
    0.7944146205,  0.7933533403,  0.7922896434,  0.7912235330, 0.7901550124,
    0.7890840848,  0.7880107536,  0.7869350220,  0.7858568932, 0.7847763705,
    0.7836934573,  0.7826081569,  0.7815204724,  0.7804304073, 0.7793379649,
    0.7782431485,  0.7771459615,  0.7760464071,  0.7749444887, 0.7738402097,
    0.7727335735,  0.7716245834,  0.7705132428,  0.7693995550, 0.7682835236,
    0.7671651518,  0.7660444431,  0.7649214009,  0.7637960286, 0.7626683297,
    0.7615383075,  0.7604059656,  0.7592713073,  0.7581343362, 0.7569950557,
    0.7558534692,  0.7547095802,  0.7535633923,  0.7524149089, 0.7512641335,
    0.7501110696,  0.7489557208,  0.7477980905,  0.7466381823, 0.7454759997,
    0.7443115462,  0.7431448255,  0.7419758410,  0.7408045963, 0.7396310950,
    0.7384553406,  0.7372773368,  0.7360970871,  0.7349145951, 0.7337298645,
    0.7325428988,  0.7313537016,  0.7301622766,  0.7289686274, 0.7277727577,
    0.7265746710,  0.7253743710,  0.7241718614,  0.7229671459, 0.7217602281,
    0.7205511117,  0.7193398003,  0.7181262978,  0.7169106077, 0.7156927337,
    0.7144726796,  0.7132504492,  0.7120260460,  0.7107994739, 0.7095707365,
    0.7083398377,  0.7071067812,  0.7058715707,  0.7046342100, 0.7033947028,
    0.7021530530,  0.7009092643,  0.6996633405,  0.6984152854, 0.6971651029,
    0.6959127966,  0.6946583705,  0.6934018283,  0.6921431739, 0.6908824111,
    0.6896195437,  0.6883545757,  0.6870875108,  0.6858183529, 0.6845471059,
    0.6832737737,  0.6819983601,  0.6807208690,  0.6794413043, 0.6781596699,
    0.6768759697,  0.6755902076,  0.6743023876,  0.6730125135, 0.6717205893,
    0.6704266190,  0.6691306064,  0.6678325555,  0.6665324702, 0.6652303547,
    0.6639262127,  0.6626200482,  0.6613118653,  0.6600016680, 0.6586894601,
    0.6573752458,  0.6560590290,  0.6547408137,  0.6534206040, 0.6520984038,
    0.6507742173,  0.6494480483,  0.6481199011,  0.6467897795, 0.6454576877,
    0.6441236298,  0.6427876097,  0.6414496316,  0.6401096995, 0.6387678175,
    0.6374239897,  0.6360782203,  0.6347305132,  0.6333808726, 0.6320293027,
    0.6306758074,  0.6293203910,  0.6279630576,  0.6266038114, 0.6252426563,
    0.6238795967,  0.6225146366,  0.6211477803,  0.6197790318, 0.6184083954,
    0.6170358751,  0.6156614753,  0.6142852001,  0.6129070537, 0.6115270402,
    0.6101451639,  0.6087614290,  0.6073758397,  0.6059884003, 0.6045991149,
    0.6032079877,  0.6018150232,  0.6004202253,  0.5990235985, 0.5976251470,
    0.5962248750,  0.5948227868,  0.5934188866,  0.5920131788, 0.5906056676,
    0.5891963574,  0.5877852523,  0.5863723567,  0.5849576750, 0.5835412114,
    0.5821229702,  0.5807029557,  0.5792811723,  0.5778576244, 0.5764323162,
    0.5750052520,  0.5735764364,  0.5721458734,  0.5707135677, 0.5692795234,
    0.5678437451,  0.5664062369,  0.5649670034,  0.5635260489, 0.5620833779,
    0.5606389946,  0.5591929035,  0.5577451090,  0.5562956155, 0.5548444274,
    0.5533915492,  0.5519369853,  0.5504807401,  0.5490228180, 0.5475632235,
    0.5461019610,  0.5446390350,  0.5431744500,  0.5417082103, 0.5402403205,
    0.5387707850,  0.5372996083,  0.5358267950,  0.5343523494, 0.5328762761,
    0.5313985795,  0.5299192642,  0.5284383347,  0.5269557955, 0.5254716511,
    0.5239859060,  0.5224985647,  0.5210096318,  0.5195191119, 0.5180270094,
    0.5165333289,  0.5150380749,  0.5135412521,  0.5120428649, 0.5105429179,
    0.5090414158,  0.5075383630,  0.5060337641,  0.5045276238, 0.5030199466,
    0.5015107372,  0.5000000000,  0.4984877398,  0.4969739610, 0.4954586684,
    0.4939418666,  0.4924235601,  0.4909037536,  0.4893824517, 0.4878596591,
    0.4863353804,  0.4848096202,  0.4832823833,  0.4817536741, 0.4802234974,
    0.4786918579,  0.4771587603,  0.4756242091,  0.4740882090, 0.4725507649,
    0.4710118812,  0.4694715628,  0.4679298143,  0.4663866403, 0.4648420457,
    0.4632960351,  0.4617486132,  0.4601997848,  0.4586495545, 0.4570979271,
    0.4555449072,  0.4539904997,  0.4524347093,  0.4508775407, 0.4493189986,
    0.4477590878,  0.4461978131,  0.4446351792,  0.4430711908, 0.4415058528,
    0.4399391699,  0.4383711468,  0.4368017884,  0.4352310994, 0.4336590846,
    0.4320857488,  0.4305110968,  0.4289351334,  0.4273578634, 0.4257792916,
    0.4241994227,  0.4226182617,  0.4210358134,  0.4194520824, 0.4178670738,
    0.4162807923,  0.4146932427,  0.4131044298,  0.4115143586, 0.4099230338,
    0.4083304604,  0.4067366431,  0.4051415868,  0.4035452964, 0.4019477767,
    0.4003490326,  0.3987490689,  0.3971478906,  0.3955455026, 0.3939419096,
    0.3923371166,  0.3907311285,  0.3891239501,  0.3875155865, 0.3859060423,
    0.3842953227,  0.3826834324,  0.3810703764,  0.3794561595, 0.3778407868,
    0.3762242631,  0.3746065934,  0.3729877826,  0.3713678356, 0.3697467573,
    0.3681245527,  0.3665012267,  0.3648767843,  0.3632512305, 0.3616245701,
    0.3599968081,  0.3583679495,  0.3567379993,  0.3551069624, 0.3534748438,
    0.3518416484,  0.3502073813,  0.3485720473,  0.3469356516, 0.3452981990,
    0.3436596946,  0.3420201433,  0.3403795502,  0.3387379202, 0.3370952584,
    0.3354515698,  0.3338068592,  0.3321611319,  0.3305143927, 0.3288666467,
    0.3272178990,  0.3255681545,  0.3239174182,  0.3222656952, 0.3206129906,
    0.3189593093,  0.3173046564,  0.3156490369,  0.3139924560, 0.3123349185,
    0.3106764296,  0.3090169944,  0.3073566178,  0.3056953050, 0.3040330609,
    0.3023698908,  0.3007057995,  0.2990407923,  0.2973748741, 0.2957080500,
    0.2940403252,  0.2923717047,  0.2907021936,  0.2890317969, 0.2873605198,
    0.2856883674,  0.2840153447,  0.2823414568,  0.2806667089, 0.2789911060,
    0.2773146533,  0.2756373558,  0.2739592187,  0.2722802470, 0.2706004460,
    0.2689198206,  0.2672383761,  0.2655561175,  0.2638730500, 0.2621891786,
    0.2605045086,  0.2588190451,  0.2571327932,  0.2554457579, 0.2537579446,
    0.2520693582,  0.2503800041,  0.2486898872,  0.2469990127, 0.2453073859,
    0.2436150118,  0.2419218956,  0.2402280425,  0.2385334576, 0.2368381461,
    0.2351421131,  0.2334453639,  0.2317479035,  0.2300497372, 0.2283508701,
    0.2266513074,  0.2249510543,  0.2232501160,  0.2215484976, 0.2198462044,
    0.2181432414,  0.2164396139,  0.2147353272,  0.2130303863, 0.2113247965,
    0.2096185629,  0.2079116908,  0.2062041854,  0.2044960518, 0.2027872954,
    0.2010779211,  0.1993679344,  0.1976573404,  0.1959461442, 0.1942343512,
    0.1925219665,  0.1908089954,  0.1890954430,  0.1873813146, 0.1856666154,
    0.1839513506,  0.1822355255,  0.1805191453,  0.1788022151, 0.1770847403,
    0.1753667261,  0.1736481777,  0.1719291003,  0.1702094992, 0.1684893796,
    0.1667687467,  0.1650476059,  0.1633259622,  0.1616038211, 0.1598811877,
    0.1581580673,  0.1564344650,  0.1547103863,  0.1529858363, 0.1512608202,
    0.1495353434,  0.1478094111,  0.1460830286,  0.1443562010, 0.1426289337,
    0.1409012319,  0.1391731010,  0.1374445460,  0.1357155724, 0.1339861854,
    0.1322563903,  0.1305261922,  0.1287955966,  0.1270646086, 0.1253332336,
    0.1236014767,  0.1218693434,  0.1201368388,  0.1184039683, 0.1166707371,
    0.1149371505,  0.1132032138,  0.1114689322,  0.1097343111, 0.1079993557,
    0.1062640713,  0.1045284633,  0.1027925368,  0.1010562972, 0.0993197497,
    0.0975828998,  0.0958457525,  0.0941083133,  0.0923705874, 0.0906325802,
    0.0888942969,  0.0871557427,  0.0854169231,  0.0836778433, 0.0819385086,
    0.0801989243,  0.0784590957,  0.0767190281,  0.0749787268, 0.0732381971,
    0.0714974443,  0.0697564737,  0.0680152907,  0.0662739004, 0.0645323083,
    0.0627905195,  0.0610485395,  0.0593063736,  0.0575640270, 0.0558215050,
    0.0540788130,  0.0523359562,  0.0505929401,  0.0488497698, 0.0471064507,
    0.0453629881,  0.0436193874,  0.0418756537,  0.0401317925, 0.0383878091,
    0.0366437087,  0.0348994967,  0.0331551784,  0.0314107591, 0.0296662441,
    0.0279216387,  0.0261769483,  0.0244321782,  0.0226873336, 0.0209424199,
    0.0191974424,  0.0174524064,  0.0157073173,  0.0139621803, 0.0122170008,
    0.0104717841,  0.0087265355,  0.0069812603,  0.0052359638, 0.0034906514,
    0.0017453284,  0.0000000000,  -0.0017453284, -0.0034906514,-0.0052359638,
    -0.0069812603, -0.0087265355, -0.0104717841, -0.0122170008,-0.0139621803,
    -0.0157073173, -0.0174524064, -0.0191974424, -0.0209424199,-0.0226873336,
    -0.0244321782, -0.0261769483, -0.0279216387, -0.0296662441,-0.0314107591,
    -0.0331551784, -0.0348994967, -0.0366437087, -0.0383878091,-0.0401317925,
    -0.0418756537, -0.0436193874, -0.0453629881, -0.0471064507,-0.0488497698,
    -0.0505929401, -0.0523359562, -0.0540788130, -0.0558215050,-0.0575640270,
    -0.0593063736, -0.0610485395, -0.0627905195, -0.0645323083,-0.0662739004,
    -0.0680152907, -0.0697564737, -0.0714974443, -0.0732381971,-0.0749787268,
    -0.0767190281, -0.0784590957, -0.0801989243, -0.0819385086,-0.0836778433,
    -0.0854169231, -0.0871557427, -0.0888942969, -0.0906325802,-0.0923705874,
    -0.0941083133, -0.0958457525, -0.0975828998, -0.0993197497,-0.1010562972,
    -0.1027925368, -0.1045284633, -0.1062640713, -0.1079993557,-0.1097343111,
    -0.1114689322, -0.1132032138, -0.1149371505, -0.1166707371,-0.1184039683,
    -0.1201368388, -0.1218693434, -0.1236014767, -0.1253332336,-0.1270646086,
    -0.1287955966, -0.1305261922, -0.1322563903, -0.1339861854,-0.1357155724,
    -0.1374445460, -0.1391731010, -0.1409012319, -0.1426289337,-0.1443562010,
    -0.1460830286, -0.1478094111, -0.1495353434, -0.1512608202,-0.1529858363,
    -0.1547103863, -0.1564344650, -0.1581580673, -0.1598811877,-0.1616038211,
    -0.1633259622, -0.1650476059, -0.1667687467, -0.1684893796,-0.1702094992,
    -0.1719291003, -0.1736481777, -0.1753667261, -0.1770847403,-0.1788022151,
    -0.1805191453, -0.1822355255, -0.1839513506, -0.1856666154,-0.1873813146,
    -0.1890954430, -0.1908089954, -0.1925219665, -0.1942343512,-0.1959461442,
    -0.1976573404, -0.1993679344, -0.2010779211, -0.2027872954,-0.2044960518,
    -0.2062041854, -0.2079116908, -0.2096185629, -0.2113247965,-0.2130303863,
    -0.2147353272, -0.2164396139, -0.2181432414, -0.2198462044,-0.2215484976,
    -0.2232501160, -0.2249510543, -0.2266513074, -0.2283508701,-0.2300497372,
    -0.2317479035, -0.2334453639, -0.2351421131, -0.2368381461,-0.2385334576,
    -0.2402280425, -0.2419218956, -0.2436150118, -0.2453073859,-0.2469990127,
    -0.2486898872, -0.2503800041, -0.2520693582, -0.2537579446,-0.2554457579,
    -0.2571327932, -0.2588190451, -0.2605045086, -0.2621891786,-0.2638730500,
    -0.2655561175, -0.2672383761, -0.2689198206, -0.2706004460,-0.2722802470,
    -0.2739592187, -0.2756373558, -0.2773146533, -0.2789911060,-0.2806667089,
    -0.2823414568, -0.2840153447, -0.2856883674, -0.2873605198,-0.2890317969,
    -0.2907021936, -0.2923717047, -0.2940403252, -0.2957080500,-0.2973748741,
    -0.2990407923, -0.3007057995, -0.3023698908, -0.3040330609,-0.3056953050,
    -0.3073566178, -0.3090169944, -0.3106764296, -0.3123349185,-0.3139924560,
    -0.3156490369, -0.3173046564, -0.3189593093, -0.3206129906,-0.3222656952,
    -0.3239174182, -0.3255681545, -0.3272178990, -0.3288666467,-0.3305143927,
    -0.3321611319, -0.3338068592, -0.3354515698, -0.3370952584,-0.3387379202,
    -0.3403795502, -0.3420201433, -0.3436596946, -0.3452981990,-0.3469356516,
    -0.3485720473, -0.3502073813, -0.3518416484, -0.3534748438,-0.3551069624,
    -0.3567379993, -0.3583679495, -0.3599968081, -0.3616245701,-0.3632512305,
    -0.3648767843, -0.3665012267, -0.3681245527, -0.3697467573,-0.3713678356,
    -0.3729877826, -0.3746065934, -0.3762242631, -0.3778407868,-0.3794561595,
    -0.3810703764, -0.3826834324, -0.3842953227, -0.3859060423,-0.3875155865,
    -0.3891239501, -0.3907311285, -0.3923371166, -0.3939419096,-0.3955455026,
    -0.3971478906, -0.3987490689, -0.4003490326, -0.4019477767,-0.4035452964,
    -0.4051415868, -0.4067366431, -0.4083304604, -0.4099230338,-0.4115143586,
    -0.4131044298, -0.4146932427, -0.4162807923, -0.4178670738,-0.4194520824,
    -0.4210358134, -0.4226182617, -0.4241994227, -0.4257792916,-0.4273578634,
    -0.4289351334, -0.4305110968, -0.4320857488, -0.4336590846,-0.4352310994,
    -0.4368017884, -0.4383711468, -0.4399391699, -0.4415058528,-0.4430711908,
    -0.4446351792, -0.4461978131, -0.4477590878, -0.4493189986,-0.4508775407,
    -0.4524347093, -0.4539904997, -0.4555449072, -0.4570979271,-0.4586495545,
    -0.4601997848, -0.4617486132, -0.4632960351, -0.4648420457,-0.4663866403,
    -0.4679298143, -0.4694715628, -0.4710118812, -0.4725507649,-0.4740882090,
    -0.4756242091, -0.4771587603, -0.4786918579, -0.4802234974,-0.4817536741,
    -0.4832823833, -0.4848096202, -0.4863353804, -0.4878596591,-0.4893824517,
    -0.4909037536, -0.4924235601, -0.4939418666, -0.4954586684,-0.4969739610,
    -0.4984877398, -0.5000000000, -0.5015107372, -0.5030199466,-0.5045276238,
    -0.5060337641, -0.5075383630, -0.5090414158, -0.5105429179,-0.5120428649,
    -0.5135412521, -0.5150380749, -0.5165333289, -0.5180270094,-0.5195191119,
    -0.5210096318, -0.5224985647, -0.5239859060, -0.5254716511,-0.5269557955,
    -0.5284383347, -0.5299192642, -0.5313985795, -0.5328762761,-0.5343523494,
    -0.5358267950, -0.5372996083, -0.5387707850, -0.5402403205,-0.5417082103,
    -0.5431744500, -0.5446390350, -0.5461019610, -0.5475632235,-0.5490228180,
    -0.5504807401, -0.5519369853, -0.5533915492, -0.5548444274,-0.5562956155,
    -0.5577451090, -0.5591929035, -0.5606389946, -0.5620833779,-0.5635260489,
    -0.5649670034, -0.5664062369, -0.5678437451, -0.5692795234,-0.5707135677,
    -0.5721458734, -0.5735764364, -0.5750052520, -0.5764323162,-0.5778576244,
    -0.5792811723, -0.5807029557, -0.5821229702, -0.5835412114,-0.5849576750,
    -0.5863723567, -0.5877852523, -0.5891963574, -0.5906056676,-0.5920131788,
    -0.5934188866, -0.5948227868, -0.5962248750, -0.5976251470,-0.5990235985,
    -0.6004202253, -0.6018150232, -0.6032079877, -0.6045991149,-0.6059884003,
    -0.6073758397, -0.6087614290, -0.6101451639, -0.6115270402,-0.6129070537,
    -0.6142852001, -0.6156614753, -0.6170358751, -0.6184083954,-0.6197790318,
    -0.6211477803, -0.6225146366, -0.6238795967, -0.6252426563,-0.6266038114,
    -0.6279630576, -0.6293203910, -0.6306758074, -0.6320293027,-0.6333808726,
    -0.6347305132, -0.6360782203, -0.6374239897, -0.6387678175,-0.6401096995,
    -0.6414496316, -0.6427876097, -0.6441236298, -0.6454576877,-0.6467897795,
    -0.6481199011, -0.6494480483, -0.6507742173, -0.6520984038,-0.6534206040,
    -0.6547408137, -0.6560590290, -0.6573752458, -0.6586894601,-0.6600016680,
    -0.6613118653, -0.6626200482, -0.6639262127, -0.6652303547,-0.6665324702,
    -0.6678325555, -0.6691306064, -0.6704266190, -0.6717205893,-0.6730125135,
    -0.6743023876, -0.6755902076, -0.6768759697, -0.6781596699,-0.6794413043,
    -0.6807208690, -0.6819983601, -0.6832737737, -0.6845471059,-0.6858183529,
    -0.6870875108, -0.6883545757, -0.6896195437, -0.6908824111,-0.6921431739,
    -0.6934018283, -0.6946583705, -0.6959127966, -0.6971651029,-0.6984152854,
    -0.6996633405, -0.7009092643, -0.7021530530, -0.7033947028,-0.7046342100,
    -0.7058715707, -0.7071067812, -0.7083398377, -0.7095707365,-0.7107994739,
    -0.7120260460, -0.7132504492, -0.7144726796, -0.7156927337,-0.7169106077,
    -0.7181262978, -0.7193398003, -0.7205511117, -0.7217602281,-0.7229671459,
    -0.7241718614, -0.7253743710, -0.7265746710, -0.7277727577,-0.7289686274,
    -0.7301622766, -0.7313537016, -0.7325428988, -0.7337298645,-0.7349145951,
    -0.7360970871, -0.7372773368, -0.7384553406, -0.7396310950,-0.7408045963,
    -0.7419758410, -0.7431448255, -0.7443115462, -0.7454759997,-0.7466381823,
    -0.7477980905, -0.7489557208, -0.7501110696, -0.7512641335,-0.7524149089,
    -0.7535633923, -0.7547095802, -0.7558534692, -0.7569950557,-0.7581343362,
    -0.7592713073, -0.7604059656, -0.7615383075, -0.7626683297,-0.7637960286,
    -0.7649214009, -0.7660444431, -0.7671651518, -0.7682835236,-0.7693995550,
    -0.7705132428, -0.7716245834, -0.7727335735, -0.7738402097,-0.7749444887,
    -0.7760464071, -0.7771459615, -0.7782431485, -0.7793379649,-0.7804304073,
    -0.7815204724, -0.7826081569, -0.7836934573, -0.7847763705,-0.7858568932,
    -0.7869350220, -0.7880107536, -0.7890840848, -0.7901550124,-0.7912235330,
    -0.7922896434, -0.7933533403, -0.7944146205, -0.7954734809,-0.7965299180,
    -0.7975839288, -0.7986355100, -0.7996846585, -0.8007313709,-0.8017756442,
    -0.8028174752, -0.8038568606, -0.8048937974, -0.8059282822,-0.8069603121,
    -0.8079898839, -0.8090169944, -0.8100416404, -0.8110638190,-0.8120835269,
    -0.8131007610, -0.8141155184, -0.8151277957, -0.8161375901,-0.8171448983,
    -0.8181497174, -0.8191520443, -0.8201518759, -0.8211492091,-0.8221440410,
    -0.8231363685, -0.8241261886, -0.8251134983, -0.8260982945,-0.8270805743,
    -0.8280603346, -0.8290375726, -0.8300122851, -0.8309844693,-0.8319541221,
    -0.8329212407, -0.8338858221, -0.8348478633, -0.8358073614,-0.8367643135,
    -0.8377187166, -0.8386705679, -0.8396198645, -0.8405666035,-0.8415107819,
    -0.8424523970, -0.8433914458, -0.8443279255, -0.8452618332,-0.8461931661,
    -0.8471219214, -0.8480480962, -0.8489716876, -0.8498926930,-0.8508111094,
    -0.8517269341, -0.8526401644, -0.8535507973, -0.8544588301,-0.8553642602,
    -0.8562670846, -0.8571673007, -0.8580649057, -0.8589598969,-0.8598522716,
    -0.8607420270, -0.8616291604, -0.8625136692, -0.8633955506,-0.8642748020,
    -0.8651514206, -0.8660254038, -0.8668967489, -0.8677654534,-0.8686315144,
    -0.8694949295, -0.8703556959, -0.8712138111, -0.8720692724,-0.8729220773,
    -0.8737722230, -0.8746197071, -0.8754645270, -0.8763066800,-0.8771461637,
    -0.8779829754, -0.8788171127, -0.8796485729, -0.8804773535,-0.8813034521,
    -0.8821268660, -0.8829475929, -0.8837656301, -0.8845809752,-0.8853936258,
    -0.8862035792, -0.8870108332, -0.8878153851, -0.8886172327,-0.8894163733,
    -0.8902128046, -0.8910065242, -0.8917975296, -0.8925858185,-0.8933713883,
    -0.8941542368, -0.8949343616, -0.8957117602, -0.8964864304,-0.8972583697,
    -0.8980275758, -0.8987940463, -0.8995577790, -0.9003187714,-0.9010770213,
    -0.9018325264, -0.9025852843, -0.9033352929, -0.9040825497,-0.9048270525,
    -0.9055687990, -0.9063077870, -0.9070440143, -0.9077774785,-0.9085081775,
    -0.9092361090, -0.9099612709, -0.9106836608, -0.9114032766,-0.9121201162,
    -0.9128341772, -0.9135454576, -0.9142539552, -0.9149596678,-0.9156625933,
    -0.9163627296, -0.9170600744, -0.9177546257, -0.9184463813,-0.9191353393,
    -0.9198214973, -0.9205048535, -0.9211854056, -0.9218631516,-0.9225380895,
    -0.9232102171, -0.9238795325, -0.9245460336, -0.9252097184,-0.9258705848,
    -0.9265286309, -0.9271838546, -0.9278362539, -0.9284858269,-0.9291325715,
    -0.9297764859, -0.9304175680, -0.9310558159, -0.9316912276,-0.9323238012,
    -0.9329535348, -0.9335804265, -0.9342044743, -0.9348256764,-0.9354440308,
    -0.9360595357, -0.9366721892, -0.9372819895, -0.9378889346,-0.9384930228,
    -0.9390942521, -0.9396926208, -0.9402881270, -0.9408807690,-0.9414705448,
    -0.9420574528, -0.9426414911, -0.9432226579, -0.9438009516,-0.9443763702,
    -0.9449489122, -0.9455185756, -0.9460853588, -0.9466492601,-0.9472102777,
    -0.9477684100, -0.9483236552, -0.9488760116, -0.9494254776,-0.9499720515,
    -0.9505157316, -0.9510565163, -0.9515944039, -0.9521293927,-0.9526614813,
    -0.9531906678, -0.9537169507, -0.9542403285, -0.9547607995,-0.9552783621,
    -0.9557930148, -0.9563047560, -0.9568135841, -0.9573194975,-0.9578224948,
    -0.9583225745, -0.9588197349, -0.9593139745, -0.9598052920,-0.9602936857,
    -0.9607791542, -0.9612616959, -0.9617413095, -0.9622179935,-0.9626917464,
    -0.9631625668, -0.9636304532, -0.9640954042, -0.9645574185,-0.9650164945,
    -0.9654726309, -0.9659258263, -0.9663760793, -0.9668233886,-0.9672677528,
    -0.9677091705, -0.9681476404, -0.9685831611, -0.9690157314,-0.9694453499,
    -0.9698720153, -0.9702957263, -0.9707164816, -0.9711342799,-0.9715491200,
    -0.9719610006, -0.9723699204, -0.9727758782, -0.9731788728,-0.9735789029,
    -0.9739759673, -0.9743700648, -0.9747611942, -0.9751493543,-0.9755345439,
    -0.9759167619, -0.9762960071, -0.9766722783, -0.9770455744,-0.9774158943,
    -0.9777832368, -0.9781476007, -0.9785089851, -0.9788673888,-0.9792228106,
    -0.9795752496, -0.9799247046, -0.9802711746, -0.9806146585,-0.9809551553,
    -0.9812926640, -0.9816271834, -0.9819587127, -0.9822872507,-0.9826127965,
    -0.9829353491, -0.9832549076, -0.9835714708, -0.9838850379,-0.9841956080,
    -0.9845031800, -0.9848077530, -0.9851093262, -0.9854078985,-0.9857034691,
    -0.9859960371, -0.9862856015, -0.9865721616, -0.9868557164,-0.9871362651,
    -0.9874138068, -0.9876883406, -0.9879598658, -0.9882283814,-0.9884938868,
    -0.9887563810, -0.9890158634, -0.9892723330, -0.9895257891,-0.9897762309,
    -0.9900236577, -0.9902680687, -0.9905094632, -0.9907478405,-0.9909831997,
    -0.9912155403, -0.9914448614, -0.9916711624, -0.9918944426,-0.9921147013,
    -0.9923319379, -0.9925461516, -0.9927573419, -0.9929655081,-0.9931706495,
    -0.9933727656, -0.9935718557, -0.9937679192, -0.9939609555,-0.9941509640,
    -0.9943379441, -0.9945218954, -0.9947028171, -0.9948807088,-0.9950555700,
    -0.9952274000, -0.9953961984, -0.9955619646, -0.9957246982,-0.9958843986,
    -0.9960410654, -0.9961946981, -0.9963452962, -0.9964928592,-0.9966373868,
    -0.9967788785, -0.9969173337, -0.9970527522, -0.9971851335,-0.9973144772,
    -0.9974407829, -0.9975640503, -0.9976842788, -0.9978014683,-0.9979156183,
    -0.9980267284, -0.9981347984, -0.9982398279, -0.9983418166,-0.9984407642,
    -0.9985366703, -0.9986295348, -0.9987193572, -0.9988061373,-0.9988898750,
    -0.9989705698, -0.9990482216, -0.9991228301, -0.9991943951,-0.9992629164,
    -0.9993283938, -0.9993908270, -0.9994502159, -0.9995065604,-0.9995598601,
    -0.9996101150, -0.9996573250, -0.9997014898, -0.9997426093,-0.9997806835,
    -0.9998157121, -0.9998476952, -0.9998766325, -0.9999025240,-0.9999253697,
    -0.9999451694, -0.9999619231, -0.9999756307, -0.9999862922,-0.9999939077,
    -0.9999984769, -1.0000000000, -0.9999984769, -0.9999939077,-0.9999862922,
    -0.9999756307, -0.9999619231, -0.9999451694, -0.9999253697,-0.9999025240,
    -0.9998766325, -0.9998476952, -0.9998157121, -0.9997806835,-0.9997426093,
    -0.9997014898, -0.9996573250, -0.9996101150, -0.9995598601,-0.9995065604,
    -0.9994502159, -0.9993908270, -0.9993283938, -0.9992629164,-0.9991943951,
    -0.9991228301, -0.9990482216, -0.9989705698, -0.9988898750,-0.9988061373,
    -0.9987193572, -0.9986295348, -0.9985366703, -0.9984407642,-0.9983418166,
    -0.9982398279, -0.9981347984, -0.9980267284, -0.9979156183,-0.9978014683,
    -0.9976842788, -0.9975640503, -0.9974407829, -0.9973144772,-0.9971851335,
    -0.9970527522, -0.9969173337, -0.9967788785, -0.9966373868,-0.9964928592,
    -0.9963452962, -0.9961946981, -0.9960410654, -0.9958843986,-0.9957246982,
    -0.9955619646, -0.9953961984, -0.9952274000, -0.9950555700,-0.9948807088,
    -0.9947028171, -0.9945218954, -0.9943379441, -0.9941509640,-0.9939609555,
    -0.9937679192, -0.9935718557, -0.9933727656, -0.9931706495,-0.9929655081,
    -0.9927573419, -0.9925461516, -0.9923319379, -0.9921147013,-0.9918944426,
    -0.9916711624, -0.9914448614, -0.9912155403, -0.9909831997,-0.9907478405,
    -0.9905094632, -0.9902680687, -0.9900236577, -0.9897762309,-0.9895257891,
    -0.9892723330, -0.9890158634, -0.9887563810, -0.9884938868,-0.9882283814,
    -0.9879598658, -0.9876883406, -0.9874138068, -0.9871362651,-0.9868557164,
    -0.9865721616, -0.9862856015, -0.9859960371, -0.9857034691,-0.9854078985,
    -0.9851093262, -0.9848077530, -0.9845031800, -0.9841956080,-0.9838850379,
    -0.9835714708, -0.9832549076, -0.9829353491, -0.9826127965,-0.9822872507,
    -0.9819587127, -0.9816271834, -0.9812926640, -0.9809551553,-0.9806146585,
    -0.9802711746, -0.9799247046, -0.9795752496, -0.9792228106,-0.9788673888,
    -0.9785089851, -0.9781476007, -0.9777832368, -0.9774158943,-0.9770455744,
    -0.9766722783, -0.9762960071, -0.9759167619, -0.9755345439,-0.9751493543,
    -0.9747611942, -0.9743700648, -0.9739759673, -0.9735789029,-0.9731788728,
    -0.9727758782, -0.9723699204, -0.9719610006, -0.9715491200,-0.9711342799,
    -0.9707164816, -0.9702957263, -0.9698720153, -0.9694453499,-0.9690157314,
    -0.9685831611, -0.9681476404, -0.9677091705, -0.9672677528,-0.9668233886,
    -0.9663760793, -0.9659258263, -0.9654726309, -0.9650164945,-0.9645574185,
    -0.9640954042, -0.9636304532, -0.9631625668, -0.9626917464,-0.9622179935,
    -0.9617413095, -0.9612616959, -0.9607791542, -0.9602936857,-0.9598052920,
    -0.9593139745, -0.9588197349, -0.9583225745, -0.9578224948,-0.9573194975,
    -0.9568135841, -0.9563047560, -0.9557930148, -0.9552783621,-0.9547607995,
    -0.9542403285, -0.9537169507, -0.9531906678, -0.9526614813,-0.9521293927,
    -0.9515944039, -0.9510565163, -0.9505157316, -0.9499720515,-0.9494254776,
    -0.9488760116, -0.9483236552, -0.9477684100, -0.9472102777,-0.9466492601,
    -0.9460853588, -0.9455185756, -0.9449489122, -0.9443763702,-0.9438009516,
    -0.9432226579, -0.9426414911, -0.9420574528, -0.9414705448,-0.9408807690,
    -0.9402881270, -0.9396926208, -0.9390942521, -0.9384930228,-0.9378889346,
    -0.9372819895, -0.9366721892, -0.9360595357, -0.9354440308,-0.9348256764,
    -0.9342044743, -0.9335804265, -0.9329535348, -0.9323238012,-0.9316912276,
    -0.9310558159, -0.9304175680, -0.9297764859, -0.9291325715,-0.9284858269,
    -0.9278362539, -0.9271838546, -0.9265286309, -0.9258705848,-0.9252097184,
    -0.9245460336, -0.9238795325, -0.9232102171, -0.9225380895,-0.9218631516,
    -0.9211854056, -0.9205048535, -0.9198214973, -0.9191353393,-0.9184463813,
    -0.9177546257, -0.9170600744, -0.9163627296, -0.9156625933,-0.9149596678,
    -0.9142539552, -0.9135454576, -0.9128341772, -0.9121201162,-0.9114032766,
    -0.9106836608, -0.9099612709, -0.9092361090, -0.9085081775,-0.9077774785,
    -0.9070440143, -0.9063077870, -0.9055687990, -0.9048270525,-0.9040825497,
    -0.9033352929, -0.9025852843, -0.9018325264, -0.9010770213,-0.9003187714,
    -0.8995577790, -0.8987940463, -0.8980275758, -0.8972583697,-0.8964864304,
    -0.8957117602, -0.8949343616, -0.8941542368, -0.8933713883,-0.8925858185,
    -0.8917975296, -0.8910065242, -0.8902128046, -0.8894163733,-0.8886172327,
    -0.8878153851, -0.8870108332, -0.8862035792, -0.8853936258,-0.8845809752,
    -0.8837656301, -0.8829475929, -0.8821268660, -0.8813034521,-0.8804773535,
    -0.8796485729, -0.8788171127, -0.8779829754, -0.8771461637,-0.8763066800,
    -0.8754645270, -0.8746197071, -0.8737722230, -0.8729220773,-0.8720692724,
    -0.8712138111, -0.8703556959, -0.8694949295, -0.8686315144,-0.8677654534,
    -0.8668967489, -0.8660254038, -0.8651514206, -0.8642748020,-0.8633955506,
    -0.8625136692, -0.8616291604, -0.8607420270, -0.8598522716,-0.8589598969,
    -0.8580649057, -0.8571673007, -0.8562670846, -0.8553642602,-0.8544588301,
    -0.8535507973, -0.8526401644, -0.8517269341, -0.8508111094,-0.8498926930,
    -0.8489716876, -0.8480480962, -0.8471219214, -0.8461931661,-0.8452618332,
    -0.8443279255, -0.8433914458, -0.8424523970, -0.8415107819,-0.8405666035,
    -0.8396198645, -0.8386705679, -0.8377187166, -0.8367643135,-0.8358073614,
    -0.8348478633, -0.8338858221, -0.8329212407, -0.8319541221,-0.8309844693,
    -0.8300122851, -0.8290375726, -0.8280603346, -0.8270805743,-0.8260982945,
    -0.8251134983, -0.8241261886, -0.8231363685, -0.8221440410,-0.8211492091,
    -0.8201518759, -0.8191520443, -0.8181497174, -0.8171448983,-0.8161375901,
    -0.8151277957, -0.8141155184, -0.8131007610, -0.8120835269,-0.8110638190,
    -0.8100416404, -0.8090169944, -0.8079898839, -0.8069603121,-0.8059282822,
    -0.8048937974, -0.8038568606, -0.8028174752, -0.8017756442,-0.8007313709,
    -0.7996846585, -0.7986355100, -0.7975839288, -0.7965299180,-0.7954734809,
    -0.7944146205, -0.7933533403, -0.7922896434, -0.7912235330,-0.7901550124,
    -0.7890840848, -0.7880107536, -0.7869350220, -0.7858568932,-0.7847763705,
    -0.7836934573, -0.7826081569, -0.7815204724, -0.7804304073,-0.7793379649,
    -0.7782431485, -0.7771459615, -0.7760464071, -0.7749444887,-0.7738402097,
    -0.7727335735, -0.7716245834, -0.7705132428, -0.7693995550,-0.7682835236,
    -0.7671651518, -0.7660444431, -0.7649214009, -0.7637960286,-0.7626683297,
    -0.7615383075, -0.7604059656, -0.7592713073, -0.7581343362,-0.7569950557,
    -0.7558534692, -0.7547095802, -0.7535633923, -0.7524149089,-0.7512641335,
    -0.7501110696, -0.7489557208, -0.7477980905, -0.7466381823,-0.7454759997,
    -0.7443115462, -0.7431448255, -0.7419758410, -0.7408045963,-0.7396310950,
    -0.7384553406, -0.7372773368, -0.7360970871, -0.7349145951,-0.7337298645,
    -0.7325428988, -0.7313537016, -0.7301622766, -0.7289686274,-0.7277727577,
    -0.7265746710, -0.7253743710, -0.7241718614, -0.7229671459,-0.7217602281,
    -0.7205511117, -0.7193398003, -0.7181262978, -0.7169106077,-0.7156927337,
    -0.7144726796, -0.7132504492, -0.7120260460, -0.7107994739,-0.7095707365,
    -0.7083398377, -0.7071067812, -0.7058715707, -0.7046342100,-0.7033947028,
    -0.7021530530, -0.7009092643, -0.6996633405, -0.6984152854,-0.6971651029,
    -0.6959127966, -0.6946583705, -0.6934018283, -0.6921431739,-0.6908824111,
    -0.6896195437, -0.6883545757, -0.6870875108, -0.6858183529,-0.6845471059,
    -0.6832737737, -0.6819983601, -0.6807208690, -0.6794413043,-0.6781596699,
    -0.6768759697, -0.6755902076, -0.6743023876, -0.6730125135,-0.6717205893,
    -0.6704266190, -0.6691306064, -0.6678325555, -0.6665324702,-0.6652303547,
    -0.6639262127, -0.6626200482, -0.6613118653, -0.6600016680,-0.6586894601,
    -0.6573752458, -0.6560590290, -0.6547408137, -0.6534206040,-0.6520984038,
    -0.6507742173, -0.6494480483, -0.6481199011, -0.6467897795,-0.6454576877,
    -0.6441236298, -0.6427876097, -0.6414496316, -0.6401096995,-0.6387678175,
    -0.6374239897, -0.6360782203, -0.6347305132, -0.6333808726,-0.6320293027,
    -0.6306758074, -0.6293203910, -0.6279630576, -0.6266038114,-0.6252426563,
    -0.6238795967, -0.6225146366, -0.6211477803, -0.6197790318,-0.6184083954,
    -0.6170358751, -0.6156614753, -0.6142852001, -0.6129070537,-0.6115270402,
    -0.6101451639, -0.6087614290, -0.6073758397, -0.6059884003,-0.6045991149,
    -0.6032079877, -0.6018150232, -0.6004202253, -0.5990235985,-0.5976251470,
    -0.5962248750, -0.5948227868, -0.5934188866, -0.5920131788,-0.5906056676,
    -0.5891963574, -0.5877852523, -0.5863723567, -0.5849576750,-0.5835412114,
    -0.5821229702, -0.5807029557, -0.5792811723, -0.5778576244,-0.5764323162,
    -0.5750052520, -0.5735764364, -0.5721458734, -0.5707135677,-0.5692795234,
    -0.5678437451, -0.5664062369, -0.5649670034, -0.5635260489,-0.5620833779,
    -0.5606389946, -0.5591929035, -0.5577451090, -0.5562956155,-0.5548444274,
    -0.5533915492, -0.5519369853, -0.5504807401, -0.5490228180,-0.5475632235,
    -0.5461019610, -0.5446390350, -0.5431744500, -0.5417082103,-0.5402403205,
    -0.5387707850, -0.5372996083, -0.5358267950, -0.5343523494,-0.5328762761,
    -0.5313985795, -0.5299192642, -0.5284383347, -0.5269557955,-0.5254716511,
    -0.5239859060, -0.5224985647, -0.5210096318, -0.5195191119,-0.5180270094,
    -0.5165333289, -0.5150380749, -0.5135412521, -0.5120428649,-0.5105429179,
    -0.5090414158, -0.5075383630, -0.5060337641, -0.5045276238,-0.5030199466,
    -0.5015107372, -0.5000000000, -0.4984877398, -0.4969739610,-0.4954586684,
    -0.4939418666, -0.4924235601, -0.4909037536, -0.4893824517,-0.4878596591,
    -0.4863353804, -0.4848096202, -0.4832823833, -0.4817536741,-0.4802234974,
    -0.4786918579, -0.4771587603, -0.4756242091, -0.4740882090,-0.4725507649,
    -0.4710118812, -0.4694715628, -0.4679298143, -0.4663866403,-0.4648420457,
    -0.4632960351, -0.4617486132, -0.4601997848, -0.4586495545,-0.4570979271,
    -0.4555449072, -0.4539904997, -0.4524347093, -0.4508775407,-0.4493189986,
    -0.4477590878, -0.4461978131, -0.4446351792, -0.4430711908,-0.4415058528,
    -0.4399391699, -0.4383711468, -0.4368017884, -0.4352310994,-0.4336590846,
    -0.4320857488, -0.4305110968, -0.4289351334, -0.4273578634,-0.4257792916,
    -0.4241994227, -0.4226182617, -0.4210358134, -0.4194520824,-0.4178670738,
    -0.4162807923, -0.4146932427, -0.4131044298, -0.4115143586,-0.4099230338,
    -0.4083304604, -0.4067366431, -0.4051415868, -0.4035452964,-0.4019477767,
    -0.4003490326, -0.3987490689, -0.3971478906, -0.3955455026,-0.3939419096,
    -0.3923371166, -0.3907311285, -0.3891239501, -0.3875155865,-0.3859060423,
    -0.3842953227, -0.3826834324, -0.3810703764, -0.3794561595,-0.3778407868,
    -0.3762242631, -0.3746065934, -0.3729877826, -0.3713678356,-0.3697467573,
    -0.3681245527, -0.3665012267, -0.3648767843, -0.3632512305,-0.3616245701,
    -0.3599968081, -0.3583679495, -0.3567379993, -0.3551069624,-0.3534748438,
    -0.3518416484, -0.3502073813, -0.3485720473, -0.3469356516,-0.3452981990,
    -0.3436596946, -0.3420201433, -0.3403795502, -0.3387379202,-0.3370952584,
    -0.3354515698, -0.3338068592, -0.3321611319, -0.3305143927,-0.3288666467,
    -0.3272178990, -0.3255681545, -0.3239174182, -0.3222656952,-0.3206129906,
    -0.3189593093, -0.3173046564, -0.3156490369, -0.3139924560,-0.3123349185,
    -0.3106764296, -0.3090169944, -0.3073566178, -0.3056953050,-0.3040330609,
    -0.3023698908, -0.3007057995, -0.2990407923, -0.2973748741,-0.2957080500,
    -0.2940403252, -0.2923717047, -0.2907021936, -0.2890317969,-0.2873605198,
    -0.2856883674, -0.2840153447, -0.2823414568, -0.2806667089,-0.2789911060,
    -0.2773146533, -0.2756373558, -0.2739592187, -0.2722802470,-0.2706004460,
    -0.2689198206, -0.2672383761, -0.2655561175, -0.2638730500,-0.2621891786,
    -0.2605045086, -0.2588190451, -0.2571327932, -0.2554457579,-0.2537579446,
    -0.2520693582, -0.2503800041, -0.2486898872, -0.2469990127,-0.2453073859,
    -0.2436150118, -0.2419218956, -0.2402280425, -0.2385334576,-0.2368381461,
    -0.2351421131, -0.2334453639, -0.2317479035, -0.2300497372,-0.2283508701,
    -0.2266513074, -0.2249510543, -0.2232501160, -0.2215484976,-0.2198462044,
    -0.2181432414, -0.2164396139, -0.2147353272, -0.2130303863,-0.2113247965,
    -0.2096185629, -0.2079116908, -0.2062041854, -0.2044960518,-0.2027872954,
    -0.2010779211, -0.1993679344, -0.1976573404, -0.1959461442,-0.1942343512,
    -0.1925219665, -0.1908089954, -0.1890954430, -0.1873813146,-0.1856666154,
    -0.1839513506, -0.1822355255, -0.1805191453, -0.1788022151,-0.1770847403,
    -0.1753667261, -0.1736481777, -0.1719291003, -0.1702094992,-0.1684893796,
    -0.1667687467, -0.1650476059, -0.1633259622, -0.1616038211,-0.1598811877,
    -0.1581580673, -0.1564344650, -0.1547103863, -0.1529858363,-0.1512608202,
    -0.1495353434, -0.1478094111, -0.1460830286, -0.1443562010,-0.1426289337,
    -0.1409012319, -0.1391731010, -0.1374445460, -0.1357155724,-0.1339861854,
    -0.1322563903, -0.1305261922, -0.1287955966, -0.1270646086,-0.1253332336,
    -0.1236014767, -0.1218693434, -0.1201368388, -0.1184039683,-0.1166707371,
    -0.1149371505, -0.1132032138, -0.1114689322, -0.1097343111,-0.1079993557,
    -0.1062640713, -0.1045284633, -0.1027925368, -0.1010562972,-0.0993197497,
    -0.0975828998, -0.0958457525, -0.0941083133, -0.0923705874,-0.0906325802,
    -0.0888942969, -0.0871557427, -0.0854169231, -0.0836778433,-0.0819385086,
    -0.0801989243, -0.0784590957, -0.0767190281, -0.0749787268,-0.0732381971,
    -0.0714974443, -0.0697564737, -0.0680152907, -0.0662739004,-0.0645323083,
    -0.0627905195, -0.0610485395, -0.0593063736, -0.0575640270,-0.0558215050,
    -0.0540788130, -0.0523359562, -0.0505929401, -0.0488497698,-0.0471064507,
    -0.0453629881, -0.0436193874, -0.0418756537, -0.0401317925,-0.0383878091,
    -0.0366437087, -0.0348994967, -0.0331551784, -0.0314107591,-0.0296662441,
    -0.0279216387, -0.0261769483, -0.0244321782, -0.0226873336,-0.0209424199,
    -0.0191974424, -0.0174524064, -0.0157073173, -0.0139621803,-0.0122170008,
    -0.0104717841, -0.0087265355, -0.0069812603, -0.0052359638,-0.0034906514,
    -0.0017453284
};

double fcosinus[3600] =
{
    1.0000000000,  0.9999984769,  0.9999939077,  0.9999862922,
    0.9999756307,  0.9999619231,  0.9999451694,  0.9999253697, 0.9999025240,
    0.9998766325,  0.9998476952,  0.9998157121,  0.9997806835, 0.9997426093,
    0.9997014898,  0.9996573250,  0.9996101150,  0.9995598601, 0.9995065604,
    0.9994502159,  0.9993908270,  0.9993283938,  0.9992629164, 0.9991943951,
    0.9991228301,  0.9990482216,  0.9989705698,  0.9988898750, 0.9988061373,
    0.9987193572,  0.9986295348,  0.9985366703,  0.9984407642, 0.9983418166,
    0.9982398279,  0.9981347984,  0.9980267284,  0.9979156183, 0.9978014683,
    0.9976842788,  0.9975640503,  0.9974407829,  0.9973144772, 0.9971851335,
    0.9970527522,  0.9969173337,  0.9967788785,  0.9966373868, 0.9964928592,
    0.9963452962,  0.9961946981,  0.9960410654,  0.9958843986, 0.9957246982,
    0.9955619646,  0.9953961984,  0.9952274000,  0.9950555700, 0.9948807088,
    0.9947028171,  0.9945218954,  0.9943379441,  0.9941509640, 0.9939609555,
    0.9937679192,  0.9935718557,  0.9933727656,  0.9931706495, 0.9929655081,
    0.9927573419,  0.9925461516,  0.9923319379,  0.9921147013, 0.9918944426,
    0.9916711624,  0.9914448614,  0.9912155403,  0.9909831997, 0.9907478405,
    0.9905094632,  0.9902680687,  0.9900236577,  0.9897762309, 0.9895257891,
    0.9892723330,  0.9890158634,  0.9887563810,  0.9884938868, 0.9882283814,
    0.9879598658,  0.9876883406,  0.9874138068,  0.9871362651, 0.9868557164,
    0.9865721616,  0.9862856015,  0.9859960371,  0.9857034691, 0.9854078985,
    0.9851093262,  0.9848077530,  0.9845031800,  0.9841956080, 0.9838850379,
    0.9835714708,  0.9832549076,  0.9829353491,  0.9826127965, 0.9822872507,
    0.9819587127,  0.9816271834,  0.9812926640,  0.9809551553, 0.9806146585,
    0.9802711746,  0.9799247046,  0.9795752496,  0.9792228106, 0.9788673888,
    0.9785089851,  0.9781476007,  0.9777832368,  0.9774158943, 0.9770455744,
    0.9766722783,  0.9762960071,  0.9759167619,  0.9755345439, 0.9751493543,
    0.9747611942,  0.9743700648,  0.9739759673,  0.9735789029, 0.9731788728,
    0.9727758782,  0.9723699204,  0.9719610006,  0.9715491200, 0.9711342799,
    0.9707164816,  0.9702957263,  0.9698720153,  0.9694453499, 0.9690157314,
    0.9685831611,  0.9681476404,  0.9677091705,  0.9672677528, 0.9668233886,
    0.9663760793,  0.9659258263,  0.9654726309,  0.9650164945, 0.9645574185,
    0.9640954042,  0.9636304532,  0.9631625668,  0.9626917464, 0.9622179935,
    0.9617413095,  0.9612616959,  0.9607791542,  0.9602936857, 0.9598052920,
    0.9593139745,  0.9588197349,  0.9583225745,  0.9578224948, 0.9573194975,
    0.9568135841,  0.9563047560,  0.9557930148,  0.9552783621, 0.9547607995,
    0.9542403285,  0.9537169507,  0.9531906678,  0.9526614813, 0.9521293927,
    0.9515944039,  0.9510565163,  0.9505157316,  0.9499720515, 0.9494254776,
    0.9488760116,  0.9483236552,  0.9477684100,  0.9472102777, 0.9466492601,
    0.9460853588,  0.9455185756,  0.9449489122,  0.9443763702, 0.9438009516,
    0.9432226579,  0.9426414911,  0.9420574528,  0.9414705448, 0.9408807690,
    0.9402881270,  0.9396926208,  0.9390942521,  0.9384930228, 0.9378889346,
    0.9372819895,  0.9366721892,  0.9360595357,  0.9354440308, 0.9348256764,
    0.9342044743,  0.9335804265,  0.9329535348,  0.9323238012, 0.9316912276,
    0.9310558159,  0.9304175680,  0.9297764859,  0.9291325715, 0.9284858269,
    0.9278362539,  0.9271838546,  0.9265286309,  0.9258705848, 0.9252097184,
    0.9245460336,  0.9238795325,  0.9232102171,  0.9225380895, 0.9218631516,
    0.9211854056,  0.9205048535,  0.9198214973,  0.9191353393, 0.9184463813,
    0.9177546257,  0.9170600744,  0.9163627296,  0.9156625933, 0.9149596678,
    0.9142539552,  0.9135454576,  0.9128341772,  0.9121201162, 0.9114032766,
    0.9106836608,  0.9099612709,  0.9092361090,  0.9085081775, 0.9077774785,
    0.9070440143,  0.9063077870,  0.9055687990,  0.9048270525, 0.9040825497,
    0.9033352929,  0.9025852843,  0.9018325264,  0.9010770213, 0.9003187714,
    0.8995577790,  0.8987940463,  0.8980275758,  0.8972583697, 0.8964864304,
    0.8957117602,  0.8949343616,  0.8941542368,  0.8933713883, 0.8925858185,
    0.8917975296,  0.8910065242,  0.8902128046,  0.8894163733, 0.8886172327,
    0.8878153851,  0.8870108332,  0.8862035792,  0.8853936258, 0.8845809752,
    0.8837656301,  0.8829475929,  0.8821268660,  0.8813034521, 0.8804773535,
    0.8796485729,  0.8788171127,  0.8779829754,  0.8771461637, 0.8763066800,
    0.8754645270,  0.8746197071,  0.8737722230,  0.8729220773, 0.8720692724,
    0.8712138111,  0.8703556959,  0.8694949295,  0.8686315144, 0.8677654534,
    0.8668967489,  0.8660254038,  0.8651514206,  0.8642748020, 0.8633955506,
    0.8625136692,  0.8616291604,  0.8607420270,  0.8598522716, 0.8589598969,
    0.8580649057,  0.8571673007,  0.8562670846,  0.8553642602, 0.8544588301,
    0.8535507973,  0.8526401644,  0.8517269341,  0.8508111094, 0.8498926930,
    0.8489716876,  0.8480480962,  0.8471219214,  0.8461931661, 0.8452618332,
    0.8443279255,  0.8433914458,  0.8424523970,  0.8415107819, 0.8405666035,
    0.8396198645,  0.8386705679,  0.8377187166,  0.8367643135, 0.8358073614,
    0.8348478633,  0.8338858221,  0.8329212407,  0.8319541221, 0.8309844693,
    0.8300122851,  0.8290375726,  0.8280603346,  0.8270805743, 0.8260982945,
    0.8251134983,  0.8241261886,  0.8231363685,  0.8221440410, 0.8211492091,
    0.8201518759,  0.8191520443,  0.8181497174,  0.8171448983, 0.8161375901,
    0.8151277957,  0.8141155184,  0.8131007610,  0.8120835269, 0.8110638190,
    0.8100416404,  0.8090169944,  0.8079898839,  0.8069603121, 0.8059282822,
    0.8048937974,  0.8038568606,  0.8028174752,  0.8017756442, 0.8007313709,
    0.7996846585,  0.7986355100,  0.7975839288,  0.7965299180, 0.7954734809,
    0.7944146205,  0.7933533403,  0.7922896434,  0.7912235330, 0.7901550124,
    0.7890840848,  0.7880107536,  0.7869350220,  0.7858568932, 0.7847763705,
    0.7836934573,  0.7826081569,  0.7815204724,  0.7804304073, 0.7793379649,
    0.7782431485,  0.7771459615,  0.7760464071,  0.7749444887, 0.7738402097,
    0.7727335735,  0.7716245834,  0.7705132428,  0.7693995550, 0.7682835236,
    0.7671651518,  0.7660444431,  0.7649214009,  0.7637960286, 0.7626683297,
    0.7615383075,  0.7604059656,  0.7592713073,  0.7581343362, 0.7569950557,
    0.7558534692,  0.7547095802,  0.7535633923,  0.7524149089, 0.7512641335,
    0.7501110696,  0.7489557208,  0.7477980905,  0.7466381823, 0.7454759997,
    0.7443115462,  0.7431448255,  0.7419758410,  0.7408045963, 0.7396310950,
    0.7384553406,  0.7372773368,  0.7360970871,  0.7349145951, 0.7337298645,
    0.7325428988,  0.7313537016,  0.7301622766,  0.7289686274, 0.7277727577,
    0.7265746710,  0.7253743710,  0.7241718614,  0.7229671459, 0.7217602281,
    0.7205511117,  0.7193398003,  0.7181262978,  0.7169106077, 0.7156927337,
    0.7144726796,  0.7132504492,  0.7120260460,  0.7107994739, 0.7095707365,
    0.7083398377,  0.7071067812,  0.7058715707,  0.7046342100, 0.7033947028,
    0.7021530530,  0.7009092643,  0.6996633405,  0.6984152854, 0.6971651029,
    0.6959127966,  0.6946583705,  0.6934018283,  0.6921431739, 0.6908824111,
    0.6896195437,  0.6883545757,  0.6870875108,  0.6858183529, 0.6845471059,
    0.6832737737,  0.6819983601,  0.6807208690,  0.6794413043, 0.6781596699,
    0.6768759697,  0.6755902076,  0.6743023876,  0.6730125135, 0.6717205893,
    0.6704266190,  0.6691306064,  0.6678325555,  0.6665324702, 0.6652303547,
    0.6639262127,  0.6626200482,  0.6613118653,  0.6600016680, 0.6586894601,
    0.6573752458,  0.6560590290,  0.6547408137,  0.6534206040, 0.6520984038,
    0.6507742173,  0.6494480483,  0.6481199011,  0.6467897795, 0.6454576877,
    0.6441236298,  0.6427876097,  0.6414496316,  0.6401096995, 0.6387678175,
    0.6374239897,  0.6360782203,  0.6347305132,  0.6333808726, 0.6320293027,
    0.6306758074,  0.6293203910,  0.6279630576,  0.6266038114, 0.6252426563,
    0.6238795967,  0.6225146366,  0.6211477803,  0.6197790318, 0.6184083954,
    0.6170358751,  0.6156614753,  0.6142852001,  0.6129070537, 0.6115270402,
    0.6101451639,  0.6087614290,  0.6073758397,  0.6059884003, 0.6045991149,
    0.6032079877,  0.6018150232,  0.6004202253,  0.5990235985, 0.5976251470,
    0.5962248750,  0.5948227868,  0.5934188866,  0.5920131788, 0.5906056676,
    0.5891963574,  0.5877852523,  0.5863723567,  0.5849576750, 0.5835412114,
    0.5821229702,  0.5807029557,  0.5792811723,  0.5778576244, 0.5764323162,
    0.5750052520,  0.5735764364,  0.5721458734,  0.5707135677, 0.5692795234,
    0.5678437451,  0.5664062369,  0.5649670034,  0.5635260489, 0.5620833779,
    0.5606389946,  0.5591929035,  0.5577451090,  0.5562956155, 0.5548444274,
    0.5533915492,  0.5519369853,  0.5504807401,  0.5490228180, 0.5475632235,
    0.5461019610,  0.5446390350,  0.5431744500,  0.5417082103, 0.5402403205,
    0.5387707850,  0.5372996083,  0.5358267950,  0.5343523494, 0.5328762761,
    0.5313985795,  0.5299192642,  0.5284383347,  0.5269557955, 0.5254716511,
    0.5239859060,  0.5224985647,  0.5210096318,  0.5195191119, 0.5180270094,
    0.5165333289,  0.5150380749,  0.5135412521,  0.5120428649, 0.5105429179,
    0.5090414158,  0.5075383630,  0.5060337641,  0.5045276238, 0.5030199466,
    0.5015107372,  0.5000000000,  0.4984877398,  0.4969739610, 0.4954586684,
    0.4939418666,  0.4924235601,  0.4909037536,  0.4893824517, 0.4878596591,
    0.4863353804,  0.4848096202,  0.4832823833,  0.4817536741, 0.4802234974,
    0.4786918579,  0.4771587603,  0.4756242091,  0.4740882090, 0.4725507649,
    0.4710118812,  0.4694715628,  0.4679298143,  0.4663866403, 0.4648420457,
    0.4632960351,  0.4617486132,  0.4601997848,  0.4586495545, 0.4570979271,
    0.4555449072,  0.4539904997,  0.4524347093,  0.4508775407, 0.4493189986,
    0.4477590878,  0.4461978131,  0.4446351792,  0.4430711908, 0.4415058528,
    0.4399391699,  0.4383711468,  0.4368017884,  0.4352310994, 0.4336590846,
    0.4320857488,  0.4305110968,  0.4289351334,  0.4273578634, 0.4257792916,
    0.4241994227,  0.4226182617,  0.4210358134,  0.4194520824, 0.4178670738,
    0.4162807923,  0.4146932427,  0.4131044298,  0.4115143586, 0.4099230338,
    0.4083304604,  0.4067366431,  0.4051415868,  0.4035452964, 0.4019477767,
    0.4003490326,  0.3987490689,  0.3971478906,  0.3955455026, 0.3939419096,
    0.3923371166,  0.3907311285,  0.3891239501,  0.3875155865, 0.3859060423,
    0.3842953227,  0.3826834324,  0.3810703764,  0.3794561595, 0.3778407868,
    0.3762242631,  0.3746065934,  0.3729877826,  0.3713678356, 0.3697467573,
    0.3681245527,  0.3665012267,  0.3648767843,  0.3632512305, 0.3616245701,
    0.3599968081,  0.3583679495,  0.3567379993,  0.3551069624, 0.3534748438,
    0.3518416484,  0.3502073813,  0.3485720473,  0.3469356516, 0.3452981990,
    0.3436596946,  0.3420201433,  0.3403795502,  0.3387379202, 0.3370952584,
    0.3354515698,  0.3338068592,  0.3321611319,  0.3305143927, 0.3288666467,
    0.3272178990,  0.3255681545,  0.3239174182,  0.3222656952, 0.3206129906,
    0.3189593093,  0.3173046564,  0.3156490369,  0.3139924560, 0.3123349185,
    0.3106764296,  0.3090169944,  0.3073566178,  0.3056953050, 0.3040330609,
    0.3023698908,  0.3007057995,  0.2990407923,  0.2973748741, 0.2957080500,
    0.2940403252,  0.2923717047,  0.2907021936,  0.2890317969, 0.2873605198,
    0.2856883674,  0.2840153447,  0.2823414568,  0.2806667089, 0.2789911060,
    0.2773146533,  0.2756373558,  0.2739592187,  0.2722802470, 0.2706004460,
    0.2689198206,  0.2672383761,  0.2655561175,  0.2638730500, 0.2621891786,
    0.2605045086,  0.2588190451,  0.2571327932,  0.2554457579, 0.2537579446,
    0.2520693582,  0.2503800041,  0.2486898872,  0.2469990127, 0.2453073859,
    0.2436150118,  0.2419218956,  0.2402280425,  0.2385334576, 0.2368381461,
    0.2351421131,  0.2334453639,  0.2317479035,  0.2300497372, 0.2283508701,
    0.2266513074,  0.2249510543,  0.2232501160,  0.2215484976, 0.2198462044,
    0.2181432414,  0.2164396139,  0.2147353272,  0.2130303863, 0.2113247965,
    0.2096185629,  0.2079116908,  0.2062041854,  0.2044960518, 0.2027872954,
    0.2010779211,  0.1993679344,  0.1976573404,  0.1959461442, 0.1942343512,
    0.1925219665,  0.1908089954,  0.1890954430,  0.1873813146, 0.1856666154,
    0.1839513506,  0.1822355255,  0.1805191453,  0.1788022151, 0.1770847403,
    0.1753667261,  0.1736481777,  0.1719291003,  0.1702094992, 0.1684893796,
    0.1667687467,  0.1650476059,  0.1633259622,  0.1616038211, 0.1598811877,
    0.1581580673,  0.1564344650,  0.1547103863,  0.1529858363, 0.1512608202,
    0.1495353434,  0.1478094111,  0.1460830286,  0.1443562010, 0.1426289337,
    0.1409012319,  0.1391731010,  0.1374445460,  0.1357155724, 0.1339861854,
    0.1322563903,  0.1305261922,  0.1287955966,  0.1270646086, 0.1253332336,
    0.1236014767,  0.1218693434,  0.1201368388,  0.1184039683, 0.1166707371,
    0.1149371505,  0.1132032138,  0.1114689322,  0.1097343111, 0.1079993557,
    0.1062640713,  0.1045284633,  0.1027925368,  0.1010562972, 0.0993197497,
    0.0975828998,  0.0958457525,  0.0941083133,  0.0923705874, 0.0906325802,
    0.0888942969,  0.0871557427,  0.0854169231,  0.0836778433, 0.0819385086,
    0.0801989243,  0.0784590957,  0.0767190281,  0.0749787268, 0.0732381971,
    0.0714974443,  0.0697564737,  0.0680152907,  0.0662739004, 0.0645323083,
    0.0627905195,  0.0610485395,  0.0593063736,  0.0575640270, 0.0558215050,
    0.0540788130,  0.0523359562,  0.0505929401,  0.0488497698, 0.0471064507,
    0.0453629881,  0.0436193874,  0.0418756537,  0.0401317925, 0.0383878091,
    0.0366437087,  0.0348994967,  0.0331551784,  0.0314107591, 0.0296662441,
    0.0279216387,  0.0261769483,  0.0244321782,  0.0226873336, 0.0209424199,
    0.0191974424,  0.0174524064,  0.0157073173,  0.0139621803, 0.0122170008,
    0.0104717841,  0.0087265355,  0.0069812603,  0.0052359638, 0.0034906514,
    0.0017453284,  0.0000000000,  -0.0017453284, -0.0034906514,-0.0052359638,
    -0.0069812603, -0.0087265355, -0.0104717841, -0.0122170008,-0.0139621803,
    -0.0157073173, -0.0174524064, -0.0191974424, -0.0209424199,-0.0226873336,
    -0.0244321782, -0.0261769483, -0.0279216387, -0.0296662441,-0.0314107591,
    -0.0331551784, -0.0348994967, -0.0366437087, -0.0383878091,-0.0401317925,
    -0.0418756537, -0.0436193874, -0.0453629881, -0.0471064507,-0.0488497698,
    -0.0505929401, -0.0523359562, -0.0540788130, -0.0558215050,-0.0575640270,
    -0.0593063736, -0.0610485395, -0.0627905195, -0.0645323083,-0.0662739004,
    -0.0680152907, -0.0697564737, -0.0714974443, -0.0732381971,-0.0749787268,
    -0.0767190281, -0.0784590957, -0.0801989243, -0.0819385086,-0.0836778433,
    -0.0854169231, -0.0871557427, -0.0888942969, -0.0906325802,-0.0923705874,
    -0.0941083133, -0.0958457525, -0.0975828998, -0.0993197497,-0.1010562972,
    -0.1027925368, -0.1045284633, -0.1062640713, -0.1079993557,-0.1097343111,
    -0.1114689322, -0.1132032138, -0.1149371505, -0.1166707371,-0.1184039683,
    -0.1201368388, -0.1218693434, -0.1236014767, -0.1253332336,-0.1270646086,
    -0.1287955966, -0.1305261922, -0.1322563903, -0.1339861854,-0.1357155724,
    -0.1374445460, -0.1391731010, -0.1409012319, -0.1426289337,-0.1443562010,
    -0.1460830286, -0.1478094111, -0.1495353434, -0.1512608202,-0.1529858363,
    -0.1547103863, -0.1564344650, -0.1581580673, -0.1598811877,-0.1616038211,
    -0.1633259622, -0.1650476059, -0.1667687467, -0.1684893796,-0.1702094992,
    -0.1719291003, -0.1736481777, -0.1753667261, -0.1770847403,-0.1788022151,
    -0.1805191453, -0.1822355255, -0.1839513506, -0.1856666154,-0.1873813146,
    -0.1890954430, -0.1908089954, -0.1925219665, -0.1942343512,-0.1959461442,
    -0.1976573404, -0.1993679344, -0.2010779211, -0.2027872954,-0.2044960518,
    -0.2062041854, -0.2079116908, -0.2096185629, -0.2113247965,-0.2130303863,
    -0.2147353272, -0.2164396139, -0.2181432414, -0.2198462044,-0.2215484976,
    -0.2232501160, -0.2249510543, -0.2266513074, -0.2283508701,-0.2300497372,
    -0.2317479035, -0.2334453639, -0.2351421131, -0.2368381461,-0.2385334576,
    -0.2402280425, -0.2419218956, -0.2436150118, -0.2453073859,-0.2469990127,
    -0.2486898872, -0.2503800041, -0.2520693582, -0.2537579446,-0.2554457579,
    -0.2571327932, -0.2588190451, -0.2605045086, -0.2621891786,-0.2638730500,
    -0.2655561175, -0.2672383761, -0.2689198206, -0.2706004460,-0.2722802470,
    -0.2739592187, -0.2756373558, -0.2773146533, -0.2789911060,-0.2806667089,
    -0.2823414568, -0.2840153447, -0.2856883674, -0.2873605198,-0.2890317969,
    -0.2907021936, -0.2923717047, -0.2940403252, -0.2957080500,-0.2973748741,
    -0.2990407923, -0.3007057995, -0.3023698908, -0.3040330609,-0.3056953050,
    -0.3073566178, -0.3090169944, -0.3106764296, -0.3123349185,-0.3139924560,
    -0.3156490369, -0.3173046564, -0.3189593093, -0.3206129906,-0.3222656952,
    -0.3239174182, -0.3255681545, -0.3272178990, -0.3288666467,-0.3305143927,
    -0.3321611319, -0.3338068592, -0.3354515698, -0.3370952584,-0.3387379202,
    -0.3403795502, -0.3420201433, -0.3436596946, -0.3452981990,-0.3469356516,
    -0.3485720473, -0.3502073813, -0.3518416484, -0.3534748438,-0.3551069624,
    -0.3567379993, -0.3583679495, -0.3599968081, -0.3616245701,-0.3632512305,
    -0.3648767843, -0.3665012267, -0.3681245527, -0.3697467573,-0.3713678356,
    -0.3729877826, -0.3746065934, -0.3762242631, -0.3778407868,-0.3794561595,
    -0.3810703764, -0.3826834324, -0.3842953227, -0.3859060423,-0.3875155865,
    -0.3891239501, -0.3907311285, -0.3923371166, -0.3939419096,-0.3955455026,
    -0.3971478906, -0.3987490689, -0.4003490326, -0.4019477767,-0.4035452964,
    -0.4051415868, -0.4067366431, -0.4083304604, -0.4099230338,-0.4115143586,
    -0.4131044298, -0.4146932427, -0.4162807923, -0.4178670738,-0.4194520824,
    -0.4210358134, -0.4226182617, -0.4241994227, -0.4257792916,-0.4273578634,
    -0.4289351334, -0.4305110968, -0.4320857488, -0.4336590846,-0.4352310994,
    -0.4368017884, -0.4383711468, -0.4399391699, -0.4415058528,-0.4430711908,
    -0.4446351792, -0.4461978131, -0.4477590878, -0.4493189986,-0.4508775407,
    -0.4524347093, -0.4539904997, -0.4555449072, -0.4570979271,-0.4586495545,
    -0.4601997848, -0.4617486132, -0.4632960351, -0.4648420457,-0.4663866403,
    -0.4679298143, -0.4694715628, -0.4710118812, -0.4725507649,-0.4740882090,
    -0.4756242091, -0.4771587603, -0.4786918579, -0.4802234974,-0.4817536741,
    -0.4832823833, -0.4848096202, -0.4863353804, -0.4878596591,-0.4893824517,
    -0.4909037536, -0.4924235601, -0.4939418666, -0.4954586684,-0.4969739610,
    -0.4984877398, -0.5000000000, -0.5015107372, -0.5030199466,-0.5045276238,
    -0.5060337641, -0.5075383630, -0.5090414158, -0.5105429179,-0.5120428649,
    -0.5135412521, -0.5150380749, -0.5165333289, -0.5180270094,-0.5195191119,
    -0.5210096318, -0.5224985647, -0.5239859060, -0.5254716511,-0.5269557955,
    -0.5284383347, -0.5299192642, -0.5313985795, -0.5328762761,-0.5343523494,
    -0.5358267950, -0.5372996083, -0.5387707850, -0.5402403205,-0.5417082103,
    -0.5431744500, -0.5446390350, -0.5461019610, -0.5475632235,-0.5490228180,
    -0.5504807401, -0.5519369853, -0.5533915492, -0.5548444274,-0.5562956155,
    -0.5577451090, -0.5591929035, -0.5606389946, -0.5620833779,-0.5635260489,
    -0.5649670034, -0.5664062369, -0.5678437451, -0.5692795234,-0.5707135677,
    -0.5721458734, -0.5735764364, -0.5750052520, -0.5764323162,-0.5778576244,
    -0.5792811723, -0.5807029557, -0.5821229702, -0.5835412114,-0.5849576750,
    -0.5863723567, -0.5877852523, -0.5891963574, -0.5906056676,-0.5920131788,
    -0.5934188866, -0.5948227868, -0.5962248750, -0.5976251470,-0.5990235985,
    -0.6004202253, -0.6018150232, -0.6032079877, -0.6045991149,-0.6059884003,
    -0.6073758397, -0.6087614290, -0.6101451639, -0.6115270402,-0.6129070537,
    -0.6142852001, -0.6156614753, -0.6170358751, -0.6184083954,-0.6197790318,
    -0.6211477803, -0.6225146366, -0.6238795967, -0.6252426563,-0.6266038114,
    -0.6279630576, -0.6293203910, -0.6306758074, -0.6320293027,-0.6333808726,
    -0.6347305132, -0.6360782203, -0.6374239897, -0.6387678175,-0.6401096995,
    -0.6414496316, -0.6427876097, -0.6441236298, -0.6454576877,-0.6467897795,
    -0.6481199011, -0.6494480483, -0.6507742173, -0.6520984038,-0.6534206040,
    -0.6547408137, -0.6560590290, -0.6573752458, -0.6586894601,-0.6600016680,
    -0.6613118653, -0.6626200482, -0.6639262127, -0.6652303547,-0.6665324702,
    -0.6678325555, -0.6691306064, -0.6704266190, -0.6717205893,-0.6730125135,
    -0.6743023876, -0.6755902076, -0.6768759697, -0.6781596699,-0.6794413043,
    -0.6807208690, -0.6819983601, -0.6832737737, -0.6845471059,-0.6858183529,
    -0.6870875108, -0.6883545757, -0.6896195437, -0.6908824111,-0.6921431739,
    -0.6934018283, -0.6946583705, -0.6959127966, -0.6971651029,-0.6984152854,
    -0.6996633405, -0.7009092643, -0.7021530530, -0.7033947028,-0.7046342100,
    -0.7058715707, -0.7071067812, -0.7083398377, -0.7095707365,-0.7107994739,
    -0.7120260460, -0.7132504492, -0.7144726796, -0.7156927337,-0.7169106077,
    -0.7181262978, -0.7193398003, -0.7205511117, -0.7217602281,-0.7229671459,
    -0.7241718614, -0.7253743710, -0.7265746710, -0.7277727577,-0.7289686274,
    -0.7301622766, -0.7313537016, -0.7325428988, -0.7337298645,-0.7349145951,
    -0.7360970871, -0.7372773368, -0.7384553406, -0.7396310950,-0.7408045963,
    -0.7419758410, -0.7431448255, -0.7443115462, -0.7454759997,-0.7466381823,
    -0.7477980905, -0.7489557208, -0.7501110696, -0.7512641335,-0.7524149089,
    -0.7535633923, -0.7547095802, -0.7558534692, -0.7569950557,-0.7581343362,
    -0.7592713073, -0.7604059656, -0.7615383075, -0.7626683297,-0.7637960286,
    -0.7649214009, -0.7660444431, -0.7671651518, -0.7682835236,-0.7693995550,
    -0.7705132428, -0.7716245834, -0.7727335735, -0.7738402097,-0.7749444887,
    -0.7760464071, -0.7771459615, -0.7782431485, -0.7793379649,-0.7804304073,
    -0.7815204724, -0.7826081569, -0.7836934573, -0.7847763705,-0.7858568932,
    -0.7869350220, -0.7880107536, -0.7890840848, -0.7901550124,-0.7912235330,
    -0.7922896434, -0.7933533403, -0.7944146205, -0.7954734809,-0.7965299180,
    -0.7975839288, -0.7986355100, -0.7996846585, -0.8007313709,-0.8017756442,
    -0.8028174752, -0.8038568606, -0.8048937974, -0.8059282822,-0.8069603121,
    -0.8079898839, -0.8090169944, -0.8100416404, -0.8110638190,-0.8120835269,
    -0.8131007610, -0.8141155184, -0.8151277957, -0.8161375901,-0.8171448983,
    -0.8181497174, -0.8191520443, -0.8201518759, -0.8211492091,-0.8221440410,
    -0.8231363685, -0.8241261886, -0.8251134983, -0.8260982945,-0.8270805743,
    -0.8280603346, -0.8290375726, -0.8300122851, -0.8309844693,-0.8319541221,
    -0.8329212407, -0.8338858221, -0.8348478633, -0.8358073614,-0.8367643135,
    -0.8377187166, -0.8386705679, -0.8396198645, -0.8405666035,-0.8415107819,
    -0.8424523970, -0.8433914458, -0.8443279255, -0.8452618332,-0.8461931661,
    -0.8471219214, -0.8480480962, -0.8489716876, -0.8498926930,-0.8508111094,
    -0.8517269341, -0.8526401644, -0.8535507973, -0.8544588301,-0.8553642602,
    -0.8562670846, -0.8571673007, -0.8580649057, -0.8589598969,-0.8598522716,
    -0.8607420270, -0.8616291604, -0.8625136692, -0.8633955506,-0.8642748020,
    -0.8651514206, -0.8660254038, -0.8668967489, -0.8677654534,-0.8686315144,
    -0.8694949295, -0.8703556959, -0.8712138111, -0.8720692724,-0.8729220773,
    -0.8737722230, -0.8746197071, -0.8754645270, -0.8763066800,-0.8771461637,
    -0.8779829754, -0.8788171127, -0.8796485729, -0.8804773535,-0.8813034521,
    -0.8821268660, -0.8829475929, -0.8837656301, -0.8845809752,-0.8853936258,
    -0.8862035792, -0.8870108332, -0.8878153851, -0.8886172327,-0.8894163733,
    -0.8902128046, -0.8910065242, -0.8917975296, -0.8925858185,-0.8933713883,
    -0.8941542368, -0.8949343616, -0.8957117602, -0.8964864304,-0.8972583697,
    -0.8980275758, -0.8987940463, -0.8995577790, -0.9003187714,-0.9010770213,
    -0.9018325264, -0.9025852843, -0.9033352929, -0.9040825497,-0.9048270525,
    -0.9055687990, -0.9063077870, -0.9070440143, -0.9077774785,-0.9085081775,
    -0.9092361090, -0.9099612709, -0.9106836608, -0.9114032766,-0.9121201162,
    -0.9128341772, -0.9135454576, -0.9142539552, -0.9149596678,-0.9156625933,
    -0.9163627296, -0.9170600744, -0.9177546257, -0.9184463813,-0.9191353393,
    -0.9198214973, -0.9205048535, -0.9211854056, -0.9218631516,-0.9225380895,
    -0.9232102171, -0.9238795325, -0.9245460336, -0.9252097184,-0.9258705848,
    -0.9265286309, -0.9271838546, -0.9278362539, -0.9284858269,-0.9291325715,
    -0.9297764859, -0.9304175680, -0.9310558159, -0.9316912276,-0.9323238012,
    -0.9329535348, -0.9335804265, -0.9342044743, -0.9348256764,-0.9354440308,
    -0.9360595357, -0.9366721892, -0.9372819895, -0.9378889346,-0.9384930228,
    -0.9390942521, -0.9396926208, -0.9402881270, -0.9408807690,-0.9414705448,
    -0.9420574528, -0.9426414911, -0.9432226579, -0.9438009516,-0.9443763702,
    -0.9449489122, -0.9455185756, -0.9460853588, -0.9466492601,-0.9472102777,
    -0.9477684100, -0.9483236552, -0.9488760116, -0.9494254776,-0.9499720515,
    -0.9505157316, -0.9510565163, -0.9515944039, -0.9521293927,-0.9526614813,
    -0.9531906678, -0.9537169507, -0.9542403285, -0.9547607995,-0.9552783621,
    -0.9557930148, -0.9563047560, -0.9568135841, -0.9573194975,-0.9578224948,
    -0.9583225745, -0.9588197349, -0.9593139745, -0.9598052920,-0.9602936857,
    -0.9607791542, -0.9612616959, -0.9617413095, -0.9622179935,-0.9626917464,
    -0.9631625668, -0.9636304532, -0.9640954042, -0.9645574185,-0.9650164945,
    -0.9654726309, -0.9659258263, -0.9663760793, -0.9668233886,-0.9672677528,
    -0.9677091705, -0.9681476404, -0.9685831611, -0.9690157314,-0.9694453499,
    -0.9698720153, -0.9702957263, -0.9707164816, -0.9711342799,-0.9715491200,
    -0.9719610006, -0.9723699204, -0.9727758782, -0.9731788728,-0.9735789029,
    -0.9739759673, -0.9743700648, -0.9747611942, -0.9751493543,-0.9755345439,
    -0.9759167619, -0.9762960071, -0.9766722783, -0.9770455744,-0.9774158943,
    -0.9777832368, -0.9781476007, -0.9785089851, -0.9788673888,-0.9792228106,
    -0.9795752496, -0.9799247046, -0.9802711746, -0.9806146585,-0.9809551553,
    -0.9812926640, -0.9816271834, -0.9819587127, -0.9822872507,-0.9826127965,
    -0.9829353491, -0.9832549076, -0.9835714708, -0.9838850379,-0.9841956080,
    -0.9845031800, -0.9848077530, -0.9851093262, -0.9854078985,-0.9857034691,
    -0.9859960371, -0.9862856015, -0.9865721616, -0.9868557164,-0.9871362651,
    -0.9874138068, -0.9876883406, -0.9879598658, -0.9882283814,-0.9884938868,
    -0.9887563810, -0.9890158634, -0.9892723330, -0.9895257891,-0.9897762309,
    -0.9900236577, -0.9902680687, -0.9905094632, -0.9907478405,-0.9909831997,
    -0.9912155403, -0.9914448614, -0.9916711624, -0.9918944426,-0.9921147013,
    -0.9923319379, -0.9925461516, -0.9927573419, -0.9929655081,-0.9931706495,
    -0.9933727656, -0.9935718557, -0.9937679192, -0.9939609555,-0.9941509640,
    -0.9943379441, -0.9945218954, -0.9947028171, -0.9948807088,-0.9950555700,
    -0.9952274000, -0.9953961984, -0.9955619646, -0.9957246982,-0.9958843986,
    -0.9960410654, -0.9961946981, -0.9963452962, -0.9964928592,-0.9966373868,
    -0.9967788785, -0.9969173337, -0.9970527522, -0.9971851335,-0.9973144772,
    -0.9974407829, -0.9975640503, -0.9976842788, -0.9978014683,-0.9979156183,
    -0.9980267284, -0.9981347984, -0.9982398279, -0.9983418166,-0.9984407642,
    -0.9985366703, -0.9986295348, -0.9987193572, -0.9988061373,-0.9988898750,
    -0.9989705698, -0.9990482216, -0.9991228301, -0.9991943951,-0.9992629164,
    -0.9993283938, -0.9993908270, -0.9994502159, -0.9995065604,-0.9995598601,
    -0.9996101150, -0.9996573250, -0.9997014898, -0.9997426093,-0.9997806835,
    -0.9998157121, -0.9998476952, -0.9998766325, -0.9999025240,-0.9999253697,
    -0.9999451694, -0.9999619231, -0.9999756307, -0.9999862922,-0.9999939077,
    -0.9999984769, -1.0000000000, -0.9999984769, -0.9999939077,-0.9999862922,
    -0.9999756307, -0.9999619231, -0.9999451694, -0.9999253697,-0.9999025240,
    -0.9998766325, -0.9998476952, -0.9998157121, -0.9997806835,-0.9997426093,
    -0.9997014898, -0.9996573250, -0.9996101150, -0.9995598601,-0.9995065604,
    -0.9994502159, -0.9993908270, -0.9993283938, -0.9992629164,-0.9991943951,
    -0.9991228301, -0.9990482216, -0.9989705698, -0.9988898750,-0.9988061373,
    -0.9987193572, -0.9986295348, -0.9985366703, -0.9984407642,-0.9983418166,
    -0.9982398279, -0.9981347984, -0.9980267284, -0.9979156183,-0.9978014683,
    -0.9976842788, -0.9975640503, -0.9974407829, -0.9973144772,-0.9971851335,
    -0.9970527522, -0.9969173337, -0.9967788785, -0.9966373868,-0.9964928592,
    -0.9963452962, -0.9961946981, -0.9960410654, -0.9958843986,-0.9957246982,
    -0.9955619646, -0.9953961984, -0.9952274000, -0.9950555700,-0.9948807088,
    -0.9947028171, -0.9945218954, -0.9943379441, -0.9941509640,-0.9939609555,
    -0.9937679192, -0.9935718557, -0.9933727656, -0.9931706495,-0.9929655081,
    -0.9927573419, -0.9925461516, -0.9923319379, -0.9921147013,-0.9918944426,
    -0.9916711624, -0.9914448614, -0.9912155403, -0.9909831997,-0.9907478405,
    -0.9905094632, -0.9902680687, -0.9900236577, -0.9897762309,-0.9895257891,
    -0.9892723330, -0.9890158634, -0.9887563810, -0.9884938868,-0.9882283814,
    -0.9879598658, -0.9876883406, -0.9874138068, -0.9871362651,-0.9868557164,
    -0.9865721616, -0.9862856015, -0.9859960371, -0.9857034691,-0.9854078985,
    -0.9851093262, -0.9848077530, -0.9845031800, -0.9841956080,-0.9838850379,
    -0.9835714708, -0.9832549076, -0.9829353491, -0.9826127965,-0.9822872507,
    -0.9819587127, -0.9816271834, -0.9812926640, -0.9809551553,-0.9806146585,
    -0.9802711746, -0.9799247046, -0.9795752496, -0.9792228106,-0.9788673888,
    -0.9785089851, -0.9781476007, -0.9777832368, -0.9774158943,-0.9770455744,
    -0.9766722783, -0.9762960071, -0.9759167619, -0.9755345439,-0.9751493543,
    -0.9747611942, -0.9743700648, -0.9739759673, -0.9735789029,-0.9731788728,
    -0.9727758782, -0.9723699204, -0.9719610006, -0.9715491200,-0.9711342799,
    -0.9707164816, -0.9702957263, -0.9698720153, -0.9694453499,-0.9690157314,
    -0.9685831611, -0.9681476404, -0.9677091705, -0.9672677528,-0.9668233886,
    -0.9663760793, -0.9659258263, -0.9654726309, -0.9650164945,-0.9645574185,
    -0.9640954042, -0.9636304532, -0.9631625668, -0.9626917464,-0.9622179935,
    -0.9617413095, -0.9612616959, -0.9607791542, -0.9602936857,-0.9598052920,
    -0.9593139745, -0.9588197349, -0.9583225745, -0.9578224948,-0.9573194975,
    -0.9568135841, -0.9563047560, -0.9557930148, -0.9552783621,-0.9547607995,
    -0.9542403285, -0.9537169507, -0.9531906678, -0.9526614813,-0.9521293927,
    -0.9515944039, -0.9510565163, -0.9505157316, -0.9499720515,-0.9494254776,
    -0.9488760116, -0.9483236552, -0.9477684100, -0.9472102777,-0.9466492601,
    -0.9460853588, -0.9455185756, -0.9449489122, -0.9443763702,-0.9438009516,
    -0.9432226579, -0.9426414911, -0.9420574528, -0.9414705448,-0.9408807690,
    -0.9402881270, -0.9396926208, -0.9390942521, -0.9384930228,-0.9378889346,
    -0.9372819895, -0.9366721892, -0.9360595357, -0.9354440308,-0.9348256764,
    -0.9342044743, -0.9335804265, -0.9329535348, -0.9323238012,-0.9316912276,
    -0.9310558159, -0.9304175680, -0.9297764859, -0.9291325715,-0.9284858269,
    -0.9278362539, -0.9271838546, -0.9265286309, -0.9258705848,-0.9252097184,
    -0.9245460336, -0.9238795325, -0.9232102171, -0.9225380895,-0.9218631516,
    -0.9211854056, -0.9205048535, -0.9198214973, -0.9191353393,-0.9184463813,
    -0.9177546257, -0.9170600744, -0.9163627296, -0.9156625933,-0.9149596678,
    -0.9142539552, -0.9135454576, -0.9128341772, -0.9121201162,-0.9114032766,
    -0.9106836608, -0.9099612709, -0.9092361090, -0.9085081775,-0.9077774785,
    -0.9070440143, -0.9063077870, -0.9055687990, -0.9048270525,-0.9040825497,
    -0.9033352929, -0.9025852843, -0.9018325264, -0.9010770213,-0.9003187714,
    -0.8995577790, -0.8987940463, -0.8980275758, -0.8972583697,-0.8964864304,
    -0.8957117602, -0.8949343616, -0.8941542368, -0.8933713883,-0.8925858185,
    -0.8917975296, -0.8910065242, -0.8902128046, -0.8894163733,-0.8886172327,
    -0.8878153851, -0.8870108332, -0.8862035792, -0.8853936258,-0.8845809752,
    -0.8837656301, -0.8829475929, -0.8821268660, -0.8813034521,-0.8804773535,
    -0.8796485729, -0.8788171127, -0.8779829754, -0.8771461637,-0.8763066800,
    -0.8754645270, -0.8746197071, -0.8737722230, -0.8729220773,-0.8720692724,
    -0.8712138111, -0.8703556959, -0.8694949295, -0.8686315144,-0.8677654534,
    -0.8668967489, -0.8660254038, -0.8651514206, -0.8642748020,-0.8633955506,
    -0.8625136692, -0.8616291604, -0.8607420270, -0.8598522716,-0.8589598969,
    -0.8580649057, -0.8571673007, -0.8562670846, -0.8553642602,-0.8544588301,
    -0.8535507973, -0.8526401644, -0.8517269341, -0.8508111094,-0.8498926930,
    -0.8489716876, -0.8480480962, -0.8471219214, -0.8461931661,-0.8452618332,
    -0.8443279255, -0.8433914458, -0.8424523970, -0.8415107819,-0.8405666035,
    -0.8396198645, -0.8386705679, -0.8377187166, -0.8367643135,-0.8358073614,
    -0.8348478633, -0.8338858221, -0.8329212407, -0.8319541221,-0.8309844693,
    -0.8300122851, -0.8290375726, -0.8280603346, -0.8270805743,-0.8260982945,
    -0.8251134983, -0.8241261886, -0.8231363685, -0.8221440410,-0.8211492091,
    -0.8201518759, -0.8191520443, -0.8181497174, -0.8171448983,-0.8161375901,
    -0.8151277957, -0.8141155184, -0.8131007610, -0.8120835269,-0.8110638190,
    -0.8100416404, -0.8090169944, -0.8079898839, -0.8069603121,-0.8059282822,
    -0.8048937974, -0.8038568606, -0.8028174752, -0.8017756442,-0.8007313709,
    -0.7996846585, -0.7986355100, -0.7975839288, -0.7965299180,-0.7954734809,
    -0.7944146205, -0.7933533403, -0.7922896434, -0.7912235330,-0.7901550124,
    -0.7890840848, -0.7880107536, -0.7869350220, -0.7858568932,-0.7847763705,
    -0.7836934573, -0.7826081569, -0.7815204724, -0.7804304073,-0.7793379649,
    -0.7782431485, -0.7771459615, -0.7760464071, -0.7749444887,-0.7738402097,
    -0.7727335735, -0.7716245834, -0.7705132428, -0.7693995550,-0.7682835236,
    -0.7671651518, -0.7660444431, -0.7649214009, -0.7637960286,-0.7626683297,
    -0.7615383075, -0.7604059656, -0.7592713073, -0.7581343362,-0.7569950557,
    -0.7558534692, -0.7547095802, -0.7535633923, -0.7524149089,-0.7512641335,
    -0.7501110696, -0.7489557208, -0.7477980905, -0.7466381823,-0.7454759997,
    -0.7443115462, -0.7431448255, -0.7419758410, -0.7408045963,-0.7396310950,
    -0.7384553406, -0.7372773368, -0.7360970871, -0.7349145951,-0.7337298645,
    -0.7325428988, -0.7313537016, -0.7301622766, -0.7289686274,-0.7277727577,
    -0.7265746710, -0.7253743710, -0.7241718614, -0.7229671459,-0.7217602281,
    -0.7205511117, -0.7193398003, -0.7181262978, -0.7169106077,-0.7156927337,
    -0.7144726796, -0.7132504492, -0.7120260460, -0.7107994739,-0.7095707365,
    -0.7083398377, -0.7071067812, -0.7058715707, -0.7046342100,-0.7033947028,
    -0.7021530530, -0.7009092643, -0.6996633405, -0.6984152854,-0.6971651029,
    -0.6959127966, -0.6946583705, -0.6934018283, -0.6921431739,-0.6908824111,
    -0.6896195437, -0.6883545757, -0.6870875108, -0.6858183529,-0.6845471059,
    -0.6832737737, -0.6819983601, -0.6807208690, -0.6794413043,-0.6781596699,
    -0.6768759697, -0.6755902076, -0.6743023876, -0.6730125135,-0.6717205893,
    -0.6704266190, -0.6691306064, -0.6678325555, -0.6665324702,-0.6652303547,
    -0.6639262127, -0.6626200482, -0.6613118653, -0.6600016680,-0.6586894601,
    -0.6573752458, -0.6560590290, -0.6547408137, -0.6534206040,-0.6520984038,
    -0.6507742173, -0.6494480483, -0.6481199011, -0.6467897795,-0.6454576877,
    -0.6441236298, -0.6427876097, -0.6414496316, -0.6401096995,-0.6387678175,
    -0.6374239897, -0.6360782203, -0.6347305132, -0.6333808726,-0.6320293027,
    -0.6306758074, -0.6293203910, -0.6279630576, -0.6266038114,-0.6252426563,
    -0.6238795967, -0.6225146366, -0.6211477803, -0.6197790318,-0.6184083954,
    -0.6170358751, -0.6156614753, -0.6142852001, -0.6129070537,-0.6115270402,
    -0.6101451639, -0.6087614290, -0.6073758397, -0.6059884003,-0.6045991149,
    -0.6032079877, -0.6018150232, -0.6004202253, -0.5990235985,-0.5976251470,
    -0.5962248750, -0.5948227868, -0.5934188866, -0.5920131788,-0.5906056676,
    -0.5891963574, -0.5877852523, -0.5863723567, -0.5849576750,-0.5835412114,
    -0.5821229702, -0.5807029557, -0.5792811723, -0.5778576244,-0.5764323162,
    -0.5750052520, -0.5735764364, -0.5721458734, -0.5707135677,-0.5692795234,
    -0.5678437451, -0.5664062369, -0.5649670034, -0.5635260489,-0.5620833779,
    -0.5606389946, -0.5591929035, -0.5577451090, -0.5562956155,-0.5548444274,
    -0.5533915492, -0.5519369853, -0.5504807401, -0.5490228180,-0.5475632235,
    -0.5461019610, -0.5446390350, -0.5431744500, -0.5417082103,-0.5402403205,
    -0.5387707850, -0.5372996083, -0.5358267950, -0.5343523494,-0.5328762761,
    -0.5313985795, -0.5299192642, -0.5284383347, -0.5269557955,-0.5254716511,
    -0.5239859060, -0.5224985647, -0.5210096318, -0.5195191119,-0.5180270094,
    -0.5165333289, -0.5150380749, -0.5135412521, -0.5120428649,-0.5105429179,
    -0.5090414158, -0.5075383630, -0.5060337641, -0.5045276238,-0.5030199466,
    -0.5015107372, -0.5000000000, -0.4984877398, -0.4969739610,-0.4954586684,
    -0.4939418666, -0.4924235601, -0.4909037536, -0.4893824517,-0.4878596591,
    -0.4863353804, -0.4848096202, -0.4832823833, -0.4817536741,-0.4802234974,
    -0.4786918579, -0.4771587603, -0.4756242091, -0.4740882090,-0.4725507649,
    -0.4710118812, -0.4694715628, -0.4679298143, -0.4663866403,-0.4648420457,
    -0.4632960351, -0.4617486132, -0.4601997848, -0.4586495545,-0.4570979271,
    -0.4555449072, -0.4539904997, -0.4524347093, -0.4508775407,-0.4493189986,
    -0.4477590878, -0.4461978131, -0.4446351792, -0.4430711908,-0.4415058528,
    -0.4399391699, -0.4383711468, -0.4368017884, -0.4352310994,-0.4336590846,
    -0.4320857488, -0.4305110968, -0.4289351334, -0.4273578634,-0.4257792916,
    -0.4241994227, -0.4226182617, -0.4210358134, -0.4194520824,-0.4178670738,
    -0.4162807923, -0.4146932427, -0.4131044298, -0.4115143586,-0.4099230338,
    -0.4083304604, -0.4067366431, -0.4051415868, -0.4035452964,-0.4019477767,
    -0.4003490326, -0.3987490689, -0.3971478906, -0.3955455026,-0.3939419096,
    -0.3923371166, -0.3907311285, -0.3891239501, -0.3875155865,-0.3859060423,
    -0.3842953227, -0.3826834324, -0.3810703764, -0.3794561595,-0.3778407868,
    -0.3762242631, -0.3746065934, -0.3729877826, -0.3713678356,-0.3697467573,
    -0.3681245527, -0.3665012267, -0.3648767843, -0.3632512305,-0.3616245701,
    -0.3599968081, -0.3583679495, -0.3567379993, -0.3551069624,-0.3534748438,
    -0.3518416484, -0.3502073813, -0.3485720473, -0.3469356516,-0.3452981990,
    -0.3436596946, -0.3420201433, -0.3403795502, -0.3387379202,-0.3370952584,
    -0.3354515698, -0.3338068592, -0.3321611319, -0.3305143927,-0.3288666467,
    -0.3272178990, -0.3255681545, -0.3239174182, -0.3222656952,-0.3206129906,
    -0.3189593093, -0.3173046564, -0.3156490369, -0.3139924560,-0.3123349185,
    -0.3106764296, -0.3090169944, -0.3073566178, -0.3056953050,-0.3040330609,
    -0.3023698908, -0.3007057995, -0.2990407923, -0.2973748741,-0.2957080500,
    -0.2940403252, -0.2923717047, -0.2907021936, -0.2890317969,-0.2873605198,
    -0.2856883674, -0.2840153447, -0.2823414568, -0.2806667089,-0.2789911060,
    -0.2773146533, -0.2756373558, -0.2739592187, -0.2722802470,-0.2706004460,
    -0.2689198206, -0.2672383761, -0.2655561175, -0.2638730500,-0.2621891786,
    -0.2605045086, -0.2588190451, -0.2571327932, -0.2554457579,-0.2537579446,
    -0.2520693582, -0.2503800041, -0.2486898872, -0.2469990127,-0.2453073859,
    -0.2436150118, -0.2419218956, -0.2402280425, -0.2385334576,-0.2368381461,
    -0.2351421131, -0.2334453639, -0.2317479035, -0.2300497372,-0.2283508701,
    -0.2266513074, -0.2249510543, -0.2232501160, -0.2215484976,-0.2198462044,
    -0.2181432414, -0.2164396139, -0.2147353272, -0.2130303863,-0.2113247965,
    -0.2096185629, -0.2079116908, -0.2062041854, -0.2044960518,-0.2027872954,
    -0.2010779211, -0.1993679344, -0.1976573404, -0.1959461442,-0.1942343512,
    -0.1925219665, -0.1908089954, -0.1890954430, -0.1873813146,-0.1856666154,
    -0.1839513506, -0.1822355255, -0.1805191453, -0.1788022151,-0.1770847403,
    -0.1753667261, -0.1736481777, -0.1719291003, -0.1702094992,-0.1684893796,
    -0.1667687467, -0.1650476059, -0.1633259622, -0.1616038211,-0.1598811877,
    -0.1581580673, -0.1564344650, -0.1547103863, -0.1529858363,-0.1512608202,
    -0.1495353434, -0.1478094111, -0.1460830286, -0.1443562010,-0.1426289337,
    -0.1409012319, -0.1391731010, -0.1374445460, -0.1357155724,-0.1339861854,
    -0.1322563903, -0.1305261922, -0.1287955966, -0.1270646086,-0.1253332336,
    -0.1236014767, -0.1218693434, -0.1201368388, -0.1184039683,-0.1166707371,
    -0.1149371505, -0.1132032138, -0.1114689322, -0.1097343111,-0.1079993557,
    -0.1062640713, -0.1045284633, -0.1027925368, -0.1010562972,-0.0993197497,
    -0.0975828998, -0.0958457525, -0.0941083133, -0.0923705874,-0.0906325802,
    -0.0888942969, -0.0871557427, -0.0854169231, -0.0836778433,-0.0819385086,
    -0.0801989243, -0.0784590957, -0.0767190281, -0.0749787268,-0.0732381971,
    -0.0714974443, -0.0697564737, -0.0680152907, -0.0662739004,-0.0645323083,
    -0.0627905195, -0.0610485395, -0.0593063736, -0.0575640270,-0.0558215050,
    -0.0540788130, -0.0523359562, -0.0505929401, -0.0488497698,-0.0471064507,
    -0.0453629881, -0.0436193874, -0.0418756537, -0.0401317925,-0.0383878091,
    -0.0366437087, -0.0348994967, -0.0331551784, -0.0314107591,-0.0296662441,
    -0.0279216387, -0.0261769483, -0.0244321782, -0.0226873336,-0.0209424199,
    -0.0191974424, -0.0174524064, -0.0157073173, -0.0139621803,-0.0122170008,
    -0.0104717841, -0.0087265355, -0.0069812603, -0.0052359638,-0.0034906514,
    -0.0017453284, -0.0000000000, 0.0017453284,  0.0034906514, 0.0052359638,
    0.0069812603,  0.0087265355,  0.0104717841,  0.0122170008, 0.0139621803,
    0.0157073173,  0.0174524064,  0.0191974424,  0.0209424199, 0.0226873336,
    0.0244321782,  0.0261769483,  0.0279216387,  0.0296662441, 0.0314107591,
    0.0331551784,  0.0348994967,  0.0366437087,  0.0383878091, 0.0401317925,
    0.0418756537,  0.0436193874,  0.0453629881,  0.0471064507, 0.0488497698,
    0.0505929401,  0.0523359562,  0.0540788130,  0.0558215050, 0.0575640270,
    0.0593063736,  0.0610485395,  0.0627905195,  0.0645323083, 0.0662739004,
    0.0680152907,  0.0697564737,  0.0714974443,  0.0732381971, 0.0749787268,
    0.0767190281,  0.0784590957,  0.0801989243,  0.0819385086, 0.0836778433,
    0.0854169231,  0.0871557427,  0.0888942969,  0.0906325802, 0.0923705874,
    0.0941083133,  0.0958457525,  0.0975828998,  0.0993197497, 0.1010562972,
    0.1027925368,  0.1045284633,  0.1062640713,  0.1079993557, 0.1097343111,
    0.1114689322,  0.1132032138,  0.1149371505,  0.1166707371, 0.1184039683,
    0.1201368388,  0.1218693434,  0.1236014767,  0.1253332336, 0.1270646086,
    0.1287955966,  0.1305261922,  0.1322563903,  0.1339861854, 0.1357155724,
    0.1374445460,  0.1391731010,  0.1409012319,  0.1426289337, 0.1443562010,
    0.1460830286,  0.1478094111,  0.1495353434,  0.1512608202, 0.1529858363,
    0.1547103863,  0.1564344650,  0.1581580673,  0.1598811877, 0.1616038211,
    0.1633259622,  0.1650476059,  0.1667687467,  0.1684893796, 0.1702094992,
    0.1719291003,  0.1736481777,  0.1753667261,  0.1770847403, 0.1788022151,
    0.1805191453,  0.1822355255,  0.1839513506,  0.1856666154, 0.1873813146,
    0.1890954430,  0.1908089954,  0.1925219665,  0.1942343512, 0.1959461442,
    0.1976573404,  0.1993679344,  0.2010779211,  0.2027872954, 0.2044960518,
    0.2062041854,  0.2079116908,  0.2096185629,  0.2113247965, 0.2130303863,
    0.2147353272,  0.2164396139,  0.2181432414,  0.2198462044, 0.2215484976,
    0.2232501160,  0.2249510543,  0.2266513074,  0.2283508701, 0.2300497372,
    0.2317479035,  0.2334453639,  0.2351421131,  0.2368381461, 0.2385334576,
    0.2402280425,  0.2419218956,  0.2436150118,  0.2453073859, 0.2469990127,
    0.2486898872,  0.2503800041,  0.2520693582,  0.2537579446, 0.2554457579,
    0.2571327932,  0.2588190451,  0.2605045086,  0.2621891786, 0.2638730500,
    0.2655561175,  0.2672383761,  0.2689198206,  0.2706004460, 0.2722802470,
    0.2739592187,  0.2756373558,  0.2773146533,  0.2789911060, 0.2806667089,
    0.2823414568,  0.2840153447,  0.2856883674,  0.2873605198, 0.2890317969,
    0.2907021936,  0.2923717047,  0.2940403252,  0.2957080500, 0.2973748741,
    0.2990407923,  0.3007057995,  0.3023698908,  0.3040330609, 0.3056953050,
    0.3073566178,  0.3090169944,  0.3106764296,  0.3123349185, 0.3139924560,
    0.3156490369,  0.3173046564,  0.3189593093,  0.3206129906, 0.3222656952,
    0.3239174182,  0.3255681545,  0.3272178990,  0.3288666467, 0.3305143927,
    0.3321611319,  0.3338068592,  0.3354515698,  0.3370952584, 0.3387379202,
    0.3403795502,  0.3420201433,  0.3436596946,  0.3452981990, 0.3469356516,
    0.3485720473,  0.3502073813,  0.3518416484,  0.3534748438, 0.3551069624,
    0.3567379993,  0.3583679495,  0.3599968081,  0.3616245701, 0.3632512305,
    0.3648767843,  0.3665012267,  0.3681245527,  0.3697467573, 0.3713678356,
    0.3729877826,  0.3746065934,  0.3762242631,  0.3778407868, 0.3794561595,
    0.3810703764,  0.3826834324,  0.3842953227,  0.3859060423, 0.3875155865,
    0.3891239501,  0.3907311285,  0.3923371166,  0.3939419096, 0.3955455026,
    0.3971478906,  0.3987490689,  0.4003490326,  0.4019477767, 0.4035452964,
    0.4051415868,  0.4067366431,  0.4083304604,  0.4099230338, 0.4115143586,
    0.4131044298,  0.4146932427,  0.4162807923,  0.4178670738, 0.4194520824,
    0.4210358134,  0.4226182617,  0.4241994227,  0.4257792916, 0.4273578634,
    0.4289351334,  0.4305110968,  0.4320857488,  0.4336590846, 0.4352310994,
    0.4368017884,  0.4383711468,  0.4399391699,  0.4415058528, 0.4430711908,
    0.4446351792,  0.4461978131,  0.4477590878,  0.4493189986, 0.4508775407,
    0.4524347093,  0.4539904997,  0.4555449072,  0.4570979271, 0.4586495545,
    0.4601997848,  0.4617486132,  0.4632960351,  0.4648420457, 0.4663866403,
    0.4679298143,  0.4694715628,  0.4710118812,  0.4725507649, 0.4740882090,
    0.4756242091,  0.4771587603,  0.4786918579,  0.4802234974, 0.4817536741,
    0.4832823833,  0.4848096202,  0.4863353804,  0.4878596591, 0.4893824517,
    0.4909037536,  0.4924235601,  0.4939418666,  0.4954586684, 0.4969739610,
    0.4984877398,  0.5000000000,  0.5015107372,  0.5030199466, 0.5045276238,
    0.5060337641,  0.5075383630,  0.5090414158,  0.5105429179, 0.5120428649,
    0.5135412521,  0.5150380749,  0.5165333289,  0.5180270094, 0.5195191119,
    0.5210096318,  0.5224985647,  0.5239859060,  0.5254716511, 0.5269557955,
    0.5284383347,  0.5299192642,  0.5313985795,  0.5328762761, 0.5343523494,
    0.5358267950,  0.5372996083,  0.5387707850,  0.5402403205, 0.5417082103,
    0.5431744500,  0.5446390350,  0.5461019610,  0.5475632235, 0.5490228180,
    0.5504807401,  0.5519369853,  0.5533915492,  0.5548444274, 0.5562956155,
    0.5577451090,  0.5591929035,  0.5606389946,  0.5620833779, 0.5635260489,
    0.5649670034,  0.5664062369,  0.5678437451,  0.5692795234, 0.5707135677,
    0.5721458734,  0.5735764364,  0.5750052520,  0.5764323162, 0.5778576244,
    0.5792811723,  0.5807029557,  0.5821229702,  0.5835412114, 0.5849576750,
    0.5863723567,  0.5877852523,  0.5891963574,  0.5906056676, 0.5920131788,
    0.5934188866,  0.5948227868,  0.5962248750,  0.5976251470, 0.5990235985,
    0.6004202253,  0.6018150232,  0.6032079877,  0.6045991149, 0.6059884003,
    0.6073758397,  0.6087614290,  0.6101451639,  0.6115270402, 0.6129070537,
    0.6142852001,  0.6156614753,  0.6170358751,  0.6184083954, 0.6197790318,
    0.6211477803,  0.6225146366,  0.6238795967,  0.6252426563, 0.6266038114,
    0.6279630576,  0.6293203910,  0.6306758074,  0.6320293027, 0.6333808726,
    0.6347305132,  0.6360782203,  0.6374239897,  0.6387678175, 0.6401096995,
    0.6414496316,  0.6427876097,  0.6441236298,  0.6454576877, 0.6467897795,
    0.6481199011,  0.6494480483,  0.6507742173,  0.6520984038, 0.6534206040,
    0.6547408137,  0.6560590290,  0.6573752458,  0.6586894601, 0.6600016680,
    0.6613118653,  0.6626200482,  0.6639262127,  0.6652303547, 0.6665324702,
    0.6678325555,  0.6691306064,  0.6704266190,  0.6717205893, 0.6730125135,
    0.6743023876,  0.6755902076,  0.6768759697,  0.6781596699, 0.6794413043,
    0.6807208690,  0.6819983601,  0.6832737737,  0.6845471059, 0.6858183529,
    0.6870875108,  0.6883545757,  0.6896195437,  0.6908824111, 0.6921431739,
    0.6934018283,  0.6946583705,  0.6959127966,  0.6971651029, 0.6984152854,
    0.6996633405,  0.7009092643,  0.7021530530,  0.7033947028, 0.7046342100,
    0.7058715707,  0.7071067812,  0.7083398377,  0.7095707365, 0.7107994739,
    0.7120260460,  0.7132504492,  0.7144726796,  0.7156927337, 0.7169106077,
    0.7181262978,  0.7193398003,  0.7205511117,  0.7217602281, 0.7229671459,
    0.7241718614,  0.7253743710,  0.7265746710,  0.7277727577, 0.7289686274,
    0.7301622766,  0.7313537016,  0.7325428988,  0.7337298645, 0.7349145951,
    0.7360970871,  0.7372773368,  0.7384553406,  0.7396310950, 0.7408045963,
    0.7419758410,  0.7431448255,  0.7443115462,  0.7454759997, 0.7466381823,
    0.7477980905,  0.7489557208,  0.7501110696,  0.7512641335, 0.7524149089,
    0.7535633923,  0.7547095802,  0.7558534692,  0.7569950557, 0.7581343362,
    0.7592713073,  0.7604059656,  0.7615383075,  0.7626683297, 0.7637960286,
    0.7649214009,  0.7660444431,  0.7671651518,  0.7682835236, 0.7693995550,
    0.7705132428,  0.7716245834,  0.7727335735,  0.7738402097, 0.7749444887,
    0.7760464071,  0.7771459615,  0.7782431485,  0.7793379649, 0.7804304073,
    0.7815204724,  0.7826081569,  0.7836934573,  0.7847763705, 0.7858568932,
    0.7869350220,  0.7880107536,  0.7890840848,  0.7901550124, 0.7912235330,
    0.7922896434,  0.7933533403,  0.7944146205,  0.7954734809, 0.7965299180,
    0.7975839288,  0.7986355100,  0.7996846585,  0.8007313709, 0.8017756442,
    0.8028174752,  0.8038568606,  0.8048937974,  0.8059282822, 0.8069603121,
    0.8079898839,  0.8090169944,  0.8100416404,  0.8110638190, 0.8120835269,
    0.8131007610,  0.8141155184,  0.8151277957,  0.8161375901, 0.8171448983,
    0.8181497174,  0.8191520443,  0.8201518759,  0.8211492091, 0.8221440410,
    0.8231363685,  0.8241261886,  0.8251134983,  0.8260982945, 0.8270805743,
    0.8280603346,  0.8290375726,  0.8300122851,  0.8309844693, 0.8319541221,
    0.8329212407,  0.8338858221,  0.8348478633,  0.8358073614, 0.8367643135,
    0.8377187166,  0.8386705679,  0.8396198645,  0.8405666035, 0.8415107819,
    0.8424523970,  0.8433914458,  0.8443279255,  0.8452618332, 0.8461931661,
    0.8471219214,  0.8480480962,  0.8489716876,  0.8498926930, 0.8508111094,
    0.8517269341,  0.8526401644,  0.8535507973,  0.8544588301, 0.8553642602,
    0.8562670846,  0.8571673007,  0.8580649057,  0.8589598969, 0.8598522716,
    0.8607420270,  0.8616291604,  0.8625136692,  0.8633955506, 0.8642748020,
    0.8651514206,  0.8660254038,  0.8668967489,  0.8677654534, 0.8686315144,
    0.8694949295,  0.8703556959,  0.8712138111,  0.8720692724, 0.8729220773,
    0.8737722230,  0.8746197071,  0.8754645270,  0.8763066800, 0.8771461637,
    0.8779829754,  0.8788171127,  0.8796485729,  0.8804773535, 0.8813034521,
    0.8821268660,  0.8829475929,  0.8837656301,  0.8845809752, 0.8853936258,
    0.8862035792,  0.8870108332,  0.8878153851,  0.8886172327, 0.8894163733,
    0.8902128046,  0.8910065242,  0.8917975296,  0.8925858185, 0.8933713883,
    0.8941542368,  0.8949343616,  0.8957117602,  0.8964864304, 0.8972583697,
    0.8980275758,  0.8987940463,  0.8995577790,  0.9003187714, 0.9010770213,
    0.9018325264,  0.9025852843,  0.9033352929,  0.9040825497, 0.9048270525,
    0.9055687990,  0.9063077870,  0.9070440143,  0.9077774785, 0.9085081775,
    0.9092361090,  0.9099612709,  0.9106836608,  0.9114032766, 0.9121201162,
    0.9128341772,  0.9135454576,  0.9142539552,  0.9149596678, 0.9156625933,
    0.9163627296,  0.9170600744,  0.9177546257,  0.9184463813, 0.9191353393,
    0.9198214973,  0.9205048535,  0.9211854056,  0.9218631516, 0.9225380895,
    0.9232102171,  0.9238795325,  0.9245460336,  0.9252097184, 0.9258705848,
    0.9265286309,  0.9271838546,  0.9278362539,  0.9284858269, 0.9291325715,
    0.9297764859,  0.9304175680,  0.9310558159,  0.9316912276, 0.9323238012,
    0.9329535348,  0.9335804265,  0.9342044743,  0.9348256764, 0.9354440308,
    0.9360595357,  0.9366721892,  0.9372819895,  0.9378889346, 0.9384930228,
    0.9390942521,  0.9396926208,  0.9402881270,  0.9408807690, 0.9414705448,
    0.9420574528,  0.9426414911,  0.9432226579,  0.9438009516, 0.9443763702,
    0.9449489122,  0.9455185756,  0.9460853588,  0.9466492601, 0.9472102777,
    0.9477684100,  0.9483236552,  0.9488760116,  0.9494254776, 0.9499720515,
    0.9505157316,  0.9510565163,  0.9515944039,  0.9521293927, 0.9526614813,
    0.9531906678,  0.9537169507,  0.9542403285,  0.9547607995, 0.9552783621,
    0.9557930148,  0.9563047560,  0.9568135841,  0.9573194975, 0.9578224948,
    0.9583225745,  0.9588197349,  0.9593139745,  0.9598052920, 0.9602936857,
    0.9607791542,  0.9612616959,  0.9617413095,  0.9622179935, 0.9626917464,
    0.9631625668,  0.9636304532,  0.9640954042,  0.9645574185, 0.9650164945,
    0.9654726309,  0.9659258263,  0.9663760793,  0.9668233886, 0.9672677528,
    0.9677091705,  0.9681476404,  0.9685831611,  0.9690157314, 0.9694453499,
    0.9698720153,  0.9702957263,  0.9707164816,  0.9711342799, 0.9715491200,
    0.9719610006,  0.9723699204,  0.9727758782,  0.9731788728, 0.9735789029,
    0.9739759673,  0.9743700648,  0.9747611942,  0.9751493543, 0.9755345439,
    0.9759167619,  0.9762960071,  0.9766722783,  0.9770455744, 0.9774158943,
    0.9777832368,  0.9781476007,  0.9785089851,  0.9788673888, 0.9792228106,
    0.9795752496,  0.9799247046,  0.9802711746,  0.9806146585, 0.9809551553,
    0.9812926640,  0.9816271834,  0.9819587127,  0.9822872507, 0.9826127965,
    0.9829353491,  0.9832549076,  0.9835714708,  0.9838850379, 0.9841956080,
    0.9845031800,  0.9848077530,  0.9851093262,  0.9854078985, 0.9857034691,
    0.9859960371,  0.9862856015,  0.9865721616,  0.9868557164, 0.9871362651,
    0.9874138068,  0.9876883406,  0.9879598658,  0.9882283814, 0.9884938868,
    0.9887563810,  0.9890158634,  0.9892723330,  0.9895257891, 0.9897762309,
    0.9900236577,  0.9902680687,  0.9905094632,  0.9907478405, 0.9909831997,
    0.9912155403,  0.9914448614,  0.9916711624,  0.9918944426, 0.9921147013,
    0.9923319379,  0.9925461516,  0.9927573419,  0.9929655081, 0.9931706495,
    0.9933727656,  0.9935718557,  0.9937679192,  0.9939609555, 0.9941509640,
    0.9943379441,  0.9945218954,  0.9947028171,  0.9948807088, 0.9950555700,
    0.9952274000,  0.9953961984,  0.9955619646,  0.9957246982, 0.9958843986,
    0.9960410654,  0.9961946981,  0.9963452962,  0.9964928592, 0.9966373868,
    0.9967788785,  0.9969173337,  0.9970527522,  0.9971851335, 0.9973144772,
    0.9974407829,  0.9975640503,  0.9976842788,  0.9978014683, 0.9979156183,
    0.9980267284,  0.9981347984,  0.9982398279,  0.9983418166, 0.9984407642,
    0.9985366703,  0.9986295348,  0.9987193572,  0.9988061373, 0.9988898750,
    0.9989705698,  0.9990482216,  0.9991228301,  0.9991943951, 0.9992629164,
    0.9993283938,  0.9993908270,  0.9994502159,  0.9995065604, 0.9995598601,
    0.9996101150,  0.9996573250,  0.9997014898,  0.9997426093, 0.9997806835,
    0.9998157121,  0.9998476952,  0.9998766325,  0.9999025240, 0.9999253697,
    0.9999451694,  0.9999619231,  0.9999756307,  0.9999862922, 0.9999939077,
    0.9999984769
};
