	/*****************************************************/
	/* sdt2eed: convertisseur de librairie sdt -> eedraw */
	/*****************************************************/

#include "fctsys.h"
#include "eelibsl.h"


/* Syntaxe ORCAD LIB
{De-Compiled Library}

PREFIX
'74LS' = 'LS'
'74S' = 'S'
END

'74LS00'
'74S00'
 SheetPath '1'
{X Size =} 6	  {Y Size =} 4		{Parts per Package =} 4
L1		1	4	9  12		  IN  'I0'
L3		2	5  10  13		  IN  'I1'
R2		3	6	8  11 DOT	  OUT 'O'
T0		14  14  14  14		  PWR 'VCC'
B0		7	7	7	7		  PWR 'GND'

VECTOR
LINE	+4.0 +0.0 +0.0 +0.0
LINE	+0.0 +0.0 +0.0 +4.0
LINE	+0.0 +4.0 +4.0 +4.0
ARC		+4.0 +2.0 +0.0 -2.0 +2.0 +0.0 +2.0
ARC		+4.0 +2.0 +2.0 +0.0 +0.0 +2.0 +2.0
END

CONVERT
L1		1	4	9  12		  IN  'I0'
L3		2	5  10  13		  IN  'I1'
R2		3	6	8  11		  OUT 'O'
T0		14  14  14  14		  PWR 'VCC'
B0		7	7	7	7		  PWR 'GND'

VECTOR
LINE	+2.5 +0.0 +0.0 +0.0
LINE	+2.5 +4.0 +0.0 +4.0
ARC		+2.5 +4.0 +3.5 -2.0 +0.0 -4.0 +4.0
ARC		-2.0 +2.0 +2.0 -2.0 +2.8 +0.0 +2.8
ARC		-2.0 +2.0 +2.8 +0.0 +2.0 +2.0 +2.8
ARC		+2.5 +0.0 +3.5 +2.0 +0.0 +4.0 +4.0
CIRCLE  +0.3 +3.0 +0.3
CIRCLE  +0.3 +1.0 +0.3
END

***********************
***********************

Syntaxe EESCHEMA LIB:

EEDRAW-LIB Version 1.0
#
# 74LS00
#
DEF 74LS00 U 0 40 Y N 4 0 N
F0 "U" 0 100 60 H V
F1 "74LS00" 0 -100 60 H V
ALIAS 74S00 74ALS00
DRAW
C -270 100 30 0 2 0
C -270 -100 30 0 2 0
A -50 200 400 -298 -899 0 2 0
A -500 0 279 -1 -449 0 2 0
A -500 0 279 449 1 0 2 0
A -50 -200 400 296 901 0 2 0
P 2 0 2 0  -50 -200  -300 -200
P 2 0 2 0  -50 200  -300 200
X GND 7 -300 -200 0 U 60 60 1 2 W N
X GND 7 -300 -200 0 U 60 60 2 2 W N
X GND 7 -300 -200 0 U 60 60 3 2 W N
X GND 7 -300 -200 0 U 60 60 4 2 W N
X VCC 14 -300 200 0 D 60 60 1 2 W N
X VCC 14 -300 200 0 D 60 60 2 2 W N
X VCC 14 -300 200 0 D 60 60 3 2 W N
X VCC 14 -300 200 0 D 60 60 4 2 W N
X O 3 600 0 300 L 60 60 1 2 O
X O 6 600 0 300 L 60 60 2 2 O
X O 8 600 0 300 L 60 60 3 2 O
X O 11 600 0 300 L 60 60 4 2 O
X I1 2 -600 -100 300 R 60 60 1 2 I
X I1 5 -600 -100 300 R 60 60 2 2 I
X I1 10 -600 -100 300 R 60 60 3 2 I
X I1 13 -600 -100 300 R 60 60 4 2 I
X I0 1 -600 100 300 R 60 60 1 2 I
X I0 4 -600 100 300 R 60 60 2 2 I
X I0 9 -600 100 300 R 60 60 3 2 I
X I0 12 -600 100 300 R 60 60 4 2 I
A 100 0 200 -1 -899 0 1 0
A 100 0 200 899 1 0 1 0
P 2 0 1 0  -300 -200  100 -200
P 2 0 1 0  -300 200  -300 -200
P 2 0 1 0  100 200  -300 200
X GND 7 -300 -200 0 U 60 60 1 1 W N
X GND 7 -300 -200 0 U 60 60 2 1 W N
X GND 7 -300 -200 0 U 60 60 3 1 W N
X GND 7 -300 -200 0 U 60 60 4 1 W N
X VCC 14 -300 200 0 D 60 60 1 1 W N
X VCC 14 -300 200 0 D 60 60 2 1 W N
X VCC 14 -300 200 0 D 60 60 3 1 W N
X VCC 14 -300 200 0 D 60 60 4 1 W N
X O 3 600 0 300 L 60 60 1 1 O I
X O 6 600 0 300 L 60 60 2 1 O
X O 8 600 0 300 L 60 60 3 1 O
X O 11 600 0 300 L 60 60 4 1 O
X I1 2 -600 -100 300 R 60 60 1 1 I
X I1 5 -600 -100 300 R 60 60 2 1 I
X I1 10 -600 -100 300 R 60 60 3 1 I
X I1 13 -600 -100 300 R 60 60 4 1 I
X I0 1 -600 100 300 R 60 60 1 1 I
X I0 4 -600 100 300 R 60 60 2 1 I
X I0 9 -600 100 300 R 60 60 3 1 I
X I0 12 -600 100 300 R 60 60 4 1 I
ENDDRAW
ENDDEF
#
#End Library
*/

#define DELIMITER '\''

#define FIND_ALIAS 1
#define FIND_ROOT 0

#define SIZE_PIN_TEXT 60

/* Fonctions importees */
void SaveActiveLibrary(FILE * SaveFile, LibraryStruct *CurrentLib );
int LibraryEntryCompare(LibraryEntryStruct *LE1, LibraryEntryStruct *LE2);


/* Fonctions */
int ReadSdtData( FILE * FileIn, LibraryEntryStruct * LibEntry,char * Line, int * LineNum);
char * GetLine(FILE *File, char *Line, int *LineNum);
char * DateAndTime(char * Line);
int ReadComposant(FILE *FileIn, char *Line, int *LineNum);
int ReadDelimitedText(char * dest, char * source, int NbMaxChar );
static void ChangeSpaces(char * Text, int NewChar);
void * MyZMalloc (int nb_octets);
void MyFree (void * pt_mem);
static void InsertAlias(PriorQue ** PQ, LibraryEntryStruct *LibEntry,
								 int *NumOfParts);
int ReadVectorDescr( FILE * FileIn, LibraryEntryStruct * LibEntry,
						char * Line, int * LineNum);
int ReadPinDescr( LibraryEntryStruct * LibEntry, char * Line );

static LibraryEntryStruct *FindLibPart(char *Name, int Alias);

/* Variables */
char FileNameSdt[1024], FileNameEEDraw[1024];
FILE * FileSdt, * FileEEDraw;
LibraryStruct CurrentLib;
PriorQue *PQ = NULL;
char AliasList[1024];
int CurrentConvert;
int IsVector;
int PartPerPack;
int GridArray;

char UsageMsg[] =
	"Usage:\n\
lib2eeshema <liborcad.src> <libeeschema.lib> (avec extensions)\n\
liborcad is an Orcad Library file in ascii form (decompiled form)\n";

/*************************************/
int main(int nbarg, char **ptarg)
/*************************************/
{
int LineNum = 1;
char Line[10000];

	if( nbarg != 3 )
		{
		printf( UsageMsg); return(1);
		}

	printf( "\n*** ORCAD Libraries Converter to EESchema libs V 1.2 ***\n");

	strcpy(FileNameSdt,ptarg[1]);
	strcpy(FileNameEEDraw,ptarg[2]);

	FileSdt = fopen( FileNameSdt, "rt" );
	if( FileSdt == NULL )
		{
		printf( " %s non trouve\n", FileNameSdt);
		return(-1);
		}

	FileEEDraw = fopen( FileNameEEDraw, "wt" );
	if( FileEEDraw == NULL )
		{
		printf( " %s impossible a creer\n", FileNameEEDraw);
		return(-1);
		}

	/* Recherche du 1er composant */
	for ( Line[0] = 0; ; )
		{
		if( GetLine( FileSdt,Line, &LineNum) == NULL ) break;
		if( Line[0] != DELIMITER ) continue;
		printf(Line); break;
		}

	/* Init Librayry header */
	PQInit( &PQ);
	PQCompFunc((PQCompFuncType) LibraryEntryCompare);

	/* Lecture des composants */
	for( ; ; )
		{
		CurrentLib.NumOfParts++ ;
		if( ReadComposant( FileSdt, Line, &LineNum) == 0 ) break;
		printf( "%s", Line);
		}

	fclose(FileSdt);

	/* Sauvegarde librairie EEDRAW */
	strncpy(CurrentLib.Name,FileNameEEDraw,40);
	SaveActiveLibrary(FileEEDraw, &CurrentLib );
	fclose(FileEEDraw);
	return(0);
}

	/************************************/
	/* void * MyZMalloc (int nb_octets) */
	/************************************/
/* Routine d'allocation memoire, avec remise a zero de la zone allouee */
void * MyZMalloc (int nb_octets)
{
void * pt_mem;
char txt[60];
	pt_mem = malloc(nb_octets);
	if (pt_mem == NULL)
		{
		printf(txt,"Memoire Insuffisante pou alloc %d octets", nb_octets);
		return(pt_mem);
		}
	memset(pt_mem, 0, nb_octets);
	return(pt_mem);
}

	/*******************************/
	/* void MyFree (void * pt_mem) */
	/*******************************/

void MyFree (void * pt_mem)
{
	free(pt_mem);
}



/*********************************************************************/
int ReadDelimitedText(char * dest, char * source, int NbMaxChar )
/*********************************************************************/
/* lit et place dans dest la chaine de caractere trouvee dans source,
	delimitee par ' .
	transfere NbMaxChar max 
	retourne le nombre de codes lus dans source
	dest est termine par NULL
*/
{
int ii, jj, flag = 0;

	 for ( ii = 1, jj = 0; ii < NbMaxChar ; jj++, source++)
		{
		if ( * source == 0 ) break;	/* fin de ligne */
		if ( * source == DELIMITER )	  /* delimiteur trouve */
			{
			if ( flag )					/* Fin de texte delimite */
				{
				if( *(source+1) != DELIMITER ) break;
				* dest = * source; dest += 2; ii++;
				}
			flag = 1;					/* Marque 1er delimiteur trouve */
			}
		else if ( flag )
			{
			* dest = * source; dest++; ii++;
			}
		}
	*dest = 0;  /* Null terminaison */
	return (jj);
}
 

/*******************************************************/
char * GetLine(FILE *File, char *Line, int *LineNum)
/*******************************************************/
/* Routine de lecture de 1 ligne utile
	retourne la 1ere ligne utile lue.
	elimine lignes vides et commentaires
	incremente *LineNum a chaque ligne lue
*/
{
	do  {
		if (fgets(Line, 255, File) == NULL) return NULL;
		*LineNum += 1;
		} while (Line[0] == '#' || Line[0] == '\n' ||  Line[0] == '\r' ||
				 Line[0] == 0);

	return Line;
}

/***********************************/
char * DateAndTime(char * Line)
/***********************************/
/* Retourne la chaine de caractere donnant date+heure */
{
#ifdef TO_DOS
struct date Date;
struct time heure;

	getdate(&Date);
	gettime(&heure);
	sprintf(Line,"%d/%d/%d-%d:%d:%d", Date.da_day, Date.da_mon, Date.da_year,
			heure.ti_hour, heure.ti_min, heure.ti_sec );
#else
	*Line = 0;
#endif
	return(Line);
}


/*******************************************************/
int ReadComposant(FILE *FileIn, char *Line, int *LineNum)
/********************************************************/
{
char Text[1024], *ptr;
int alias = 0, ii, len;
LibraryEntryStruct * LibEntry;
LibraryDrawEntryStruct *Drawing;

	ReadDelimitedText(Text, Line, sizeof(LibEntry->Name) );
	/* Creation de 1 DEF/ENDDEF part entry from library: */
	LibEntry = (LibraryEntryStruct *)
						MyZMalloc(sizeof(LibraryEntryStruct));

	LibEntry->Type = ROOT;
	LibEntry->PrefixSize =  DEFAULT_SIZE_TEXT;
	LibEntry->PrefixPosY = PIN_WIDTH;
	LibEntry->NamePosY = - PIN_WIDTH;
	LibEntry->NameSize =  DEFAULT_SIZE_TEXT;
	LibEntry->Prefix[0] = 'U';
	LibEntry->Prefix[1] = 0; LibEntry->DrawPrefix = TRUE;
	LibEntry->TextInside = 30;
	LibEntry->DrawPinNum = 1;
	LibEntry->DrawPinName = 1;
	LibEntry->NumOfUnits = 1;

	/* Copy part name (avec suppression des espaces )*/
	ChangeSpaces(Text, '_');
	strupr(Text);
	LibEntry->DrawName = TRUE;
	strcpy(LibEntry->Name, Text);
	AliasList[0] = 0;

	CurrentConvert = 1, IsVector = 0;

	for ( ;; )
		{
		if( GetLine( FileIn,Line, LineNum) == NULL ) return(0);

		if( Line[0] == DELIMITER )
			{
			ReadDelimitedText(Text, Line, sizeof(LibEntry->Name) );
			for( ptr = Text; *ptr ; ptr++ ) if (*ptr == ' ') *ptr = '_';
			if (alias == 0 ) strcpy(AliasList, Text);
			else
				{
				strcat(AliasList, " "); strcat(AliasList, Text);
				}
			alias++;
			continue;
			}

		if( strncmp(Line,"REFERENCE", 9) == 0 ) /* Copy prefix. */
			{
			LibEntry->DrawPrefix = TRUE;
			ReadDelimitedText(Text, Line+9, sizeof(LibEntry->Prefix) );
			strcpy(LibEntry->Prefix, Text);
			continue;
			}

		if( strnicmp(Line," SheetPath", 9) == 0 ) /* sheet name */
			{
			LibraryFieldEntry * Fstruct;
			ReadDelimitedText(Text, Line+9, sizeof(Text)-1 );
			Fstruct = (LibraryFieldEntry *) MyZMalloc (sizeof(LibraryFieldEntry));
			Fstruct->Text = strdup(Text);
			Fstruct->Size = DEFAULT_SIZE_TEXT;
			Fstruct->FieldId = SHEET_NAME;
			Fstruct->Pnext = LibEntry->Fields;
			LibEntry->Fields = Fstruct;
			continue;
			}

		ReadSdtData( FileIn, LibEntry, Line, LineNum);

		/* Positionnement des textes Ref et Val en haut et en bas du cmp */
		LibEntry->PrefixPosY = LibEntry->BBoxMaxY + 50;
		LibEntry->NamePosY = LibEntry->BBoxMinY - 50;
		Drawing = LibEntry->Drawings;
		/* calcul de l'absisse de la REF */
		ii = LibEntry->BBoxMinX;
		for ( ;Drawing != NULL; Drawing = Drawing->Pnext )
			{
			if( Drawing->DrawType != PIN_DRAW_TYPE ) continue;
			if( Drawing->U.Pin.Orient != PIN_DOWN ) continue;
			if ( ii < Drawing->U.Pin.posX ) ii = Drawing->U.Pin.posX;
			}
		len = strlen(LibEntry->Prefix) + 3;
		LibEntry->PrefixPosX = ii + 100 + (len * LibEntry->PrefixSize/2);
		if( LibEntry->PrefixPosX < 0 ) LibEntry->PrefixPosX = 0;

		/* calcul de l'absisse de la VAL */
		ii = LibEntry->BBoxMinX;
		Drawing = LibEntry->Drawings;
		for ( ;Drawing != NULL; Drawing = Drawing->Pnext )
			{
			if( Drawing->DrawType != PIN_DRAW_TYPE ) continue;
			if( Drawing->U.Pin.Orient != PIN_UP ) continue;
			if ( ii < Drawing->U.Pin.posX ) ii = Drawing->U.Pin.posX;
			}
		len = strlen(LibEntry->Name) + 1;
		LibEntry->NamePosX = ii + 100 + (len * LibEntry->NameSize/2);
		if( LibEntry->NamePosX < 0 ) LibEntry->NamePosX = 0;

		PQInsert(&PQ, LibEntry);
		InsertAlias(&PQ, LibEntry, &CurrentLib.NumOfParts);
		if(CurrentLib.Entries == NULL) CurrentLib.Entries = PQ;
		return(1);
		}

	return(1);
}

/*************************************************************************************/
static void InsertAlias(PriorQue ** PQ, LibraryEntryStruct *LibEntry, int *NumOfParts)
/**************************************************************************************/
/* Insere les differents alias du composant en cours
*/
{
char * Name;
LibraryAliasStruct *AliasEntry;


	if(AliasList[0] == 0) return; /* Pas d'alias pour ce composant */

	LibEntry->AliasList = strdup(AliasList);

	Name = strtok(AliasList," \t\n");
	while( Name )
		{
		AliasEntry = (LibraryAliasStruct *) MyZMalloc(sizeof(LibraryAliasStruct));
		AliasEntry->Type = ALIAS;
		ChangeSpaces(Name, '_');
		strncpy(AliasEntry->Name, Name, PART_NAME_LEN);
		strncpy(AliasEntry->RootName, LibEntry->Name, PART_NAME_LEN);
		++*NumOfParts;
		PQInsert(PQ, AliasEntry);
		Name = strtok(NULL," \t\n");
		}
}

/************************************************************/
int ReadSdtData( FILE * FileIn, LibraryEntryStruct * LibEntry,
			char * Line, int * LineNum)
/************************************************************/
{
int end = 0, dx, dy;
char Buff[80], *Text, fillchar;
LibraryDrawEntryStruct *New = NULL;


	/* Read X size, Y size, Parts per Pack */
	/* suppression des commentaires */
	Text = Line; fillchar = 0;
	PartPerPack = 1;
	while ( *Text )
		{
		if( *Text == '{' ) fillchar = ' ';
		if( *Text == '}' ) { *Text = ' '; fillchar = 0; }
		if( fillchar ) *Text = fillchar;
		Text++;
		}
	sscanf(Line, " %d %d %s", &dx, &dy, Buff);
	if( (Buff[0] == 'G') || (Buff[0] == 'g') ) GridArray = 1;
	else
		{
		GridArray = 0;
		sscanf(Buff,"%d", &PartPerPack);
		}

	LibEntry->NumOfUnits = PartPerPack;
	if( LibEntry->NumOfUnits == 0 )
		{
		LibEntry->DrawPinNum = 0; LibEntry->NumOfUnits = 1;
		}

	LibEntry->BBoxMinX = - dx * 50;
	LibEntry->BBoxMinY = - dy * 50;
	LibEntry->BBoxMaxX = dx * 50;
	LibEntry->BBoxMaxY = dy * 50;

	/* Lecture des lignes suivantes : */
	for( ; end == 0; )
		{
		if( GetLine( FileIn,Line, LineNum) == NULL )
			{
			Line[0] = DELIMITER;
			end = 1;
			}

		switch( Line[0] )
			{
			case 'L':
			case 'R':
			case 'T':
			case 'B':
				ReadPinDescr( LibEntry, Line);
				break;

			case 'C':	/* Convert */
				CurrentConvert = 2;
				break;

			case 'V':
				IsVector = 1;
				LibEntry->DrawPinName = 0;
				ReadVectorDescr( FileIn, LibEntry, Line, LineNum);
				break;

			case DELIMITER:
				end = 1;
				/* Placement d'un contour si pas de contour */
				if(IsVector)
					{
					LibEntry->DrawPinName = 0;
					break;
					}

				New = (LibraryDrawEntryStruct *)
						MyZMalloc(sizeof(LibraryDrawEntryStruct));
				New->DrawType = SQUARE_DRAW_TYPE;
				New->U.Sqr.width = 0;

				New->Pnext = LibEntry->Drawings;
				LibEntry->Drawings = New;

				New->U.Sqr.x1 = LibEntry->BBoxMinX;
				New->U.Sqr.y1 = LibEntry->BBoxMinY;

				New->U.Sqr.x2 = LibEntry->BBoxMaxX;
				New->U.Sqr.y2 = LibEntry->BBoxMaxY;

				New->Unit = 0; New->Convert = CurrentConvert;
				break;

				break;
			 }
		}

	return(1);
}


/************************************************************/
/* ReadVectorDescr( FILE * FileIn, LibraryEntryStruct * LibEntry,*/
/*								char * Line, int * LineNum) */
/************************************************************/

int ReadVectorDescr( FILE * FileIn, LibraryEntryStruct * LibEntry,char * Line, int * LineNum)
{
LibraryEntryStruct *Entry;
LibraryDrawEntryStruct *New = NULL, *Drawing;
float x1, y1, x2, y2;
int * Poly;
char Buffer[1024], *Text;


	/* analyse de la ligne VECTOR ..: peut etre suivie d'un nom */
	strcpy(Buffer, Line);
	strtok(Buffer," '\n\r\t");
	Text = strtok(NULL,"'\n\r");

	if( Text && strlen(Text) ) /* Report a une description deja existante */
		{
		Entry = FindLibPart(Text, FIND_ALIAS);
		if( Entry == NULL )
			{
  			printf(" Line <%s>: %s non trouve",Line, Text); return(1);
			}
		/* copie de la description */
		for( Drawing = Entry->Drawings ; Drawing != NULL; Drawing = Drawing->Pnext)
			{
			if( Drawing->DrawType == PIN_DRAW_TYPE ) continue;
			New = (LibraryDrawEntryStruct *)
						MyZMalloc(sizeof(LibraryDrawEntryStruct));
			*New = *Drawing;
			New->Pnext = LibEntry->Drawings;
			LibEntry->Drawings = New;
			}
		return(1);
		}

	/* Lecture des lignes suivantes : */
	for( ; ; )
		{
		if( GetLine( FileIn,Line, LineNum) == NULL ) return(0);

		switch( Line[0] )
			{
			case 'L':	/* Line */
				New = (LibraryDrawEntryStruct *)
						MyZMalloc(sizeof(LibraryDrawEntryStruct));
				New->DrawType = POLYLINE_DRAW_TYPE;
				New->U.Poly.width = 0;
				New->U.Poly.n = 2;
				Poly = New->U.Poly.PolyList
					 = (int*) MyZMalloc( 4 * sizeof(int) );

				New->Pnext = LibEntry->Drawings;
				LibEntry->Drawings = New;
				sscanf(Line+4," %f %f %f %f", &x1, &y1, &x2, &y2);

				*Poly = (int)( x1 * 100 ) + LibEntry->BBoxMinX; Poly++;
				*Poly = LibEntry->BBoxMaxY - (int)( y1 * 100 ); Poly++;
				*Poly = (int)( x2 * 100 ) + LibEntry->BBoxMinX; Poly++;
				*Poly = LibEntry->BBoxMaxY - (int)( y2 * 100 );

				New->Unit = 0; New->Convert = CurrentConvert;
				break;

			case 'A':	/* Arc */
				{
				float sx, sy, ex, ey;
				New = (LibraryDrawEntryStruct *)
						MyZMalloc(sizeof(LibraryDrawEntryStruct));
				New->DrawType = ARC_DRAW_TYPE;
				New->Pnext = LibEntry->Drawings;
				LibEntry->Drawings = New;
				sscanf(Line+4," %f %f %f %f %f %f %f", &x1, &y1,
									&sx, &sy, &ex, &ey, &x2);

				New->U.Arc.x = (int)( x1 * 100 ) + LibEntry->BBoxMinX;
				New->U.Arc.y = LibEntry->BBoxMaxY - (int)( y1 * 100 ) ;
				New->U.Arc.r = (int)( x2 * 100 );
				New->U.Arc.width = 0;

				/* calcul des angles de depart et d'arrivee */
				New->U.Arc.t1 = (int)(atan2(-sy, sx) * 1800 /M_PI);
				while( New->U.Arc.t1 < 0 ) New->U.Arc.t1 += 3600;

				New->U.Arc.t2 = (int)(atan2(-ey, ex) * 1800 /M_PI);
				while( New->U.Arc.t2 < 0 ) New->U.Arc.t2 += 3600;

				New->Unit = 0; New->Convert = CurrentConvert;
				break;
				}

			case 'C':	/* Cercle */
				New = (LibraryDrawEntryStruct *)
						MyZMalloc(sizeof(LibraryDrawEntryStruct));
				New->DrawType = CIRCLE_DRAW_TYPE;
				New->Pnext = LibEntry->Drawings;
				LibEntry->Drawings = New;
				sscanf(Line+6," %f %f %f", &x1, &y1, &x2);

				New->U.Circ.x = (int)( x1 * 100 ) + LibEntry->BBoxMinX;
				New->U.Circ.y = LibEntry->BBoxMaxY - (int)( y1 * 100 ) ;
				New->U.Circ.r = (int)( x2 * 100 );
				New->U.Circ.width = 0;

				New->Unit = 0; New->Convert = CurrentConvert;
				break;

 
			case 'T':	/* Text */
				New = (LibraryDrawEntryStruct *)
						MyZMalloc(sizeof(LibraryDrawEntryStruct));
				New->DrawType = TEXT_DRAW_TYPE;
				New->Pnext = LibEntry->Drawings;
				LibEntry->Drawings = New;
				sscanf(Line+4," %f %f %f", &x1, &y1, &x2);

				New->U.Text.x = (int)( x1 * 100 ) + LibEntry->BBoxMinX;
				New->U.Text.y = LibEntry->BBoxMaxY - (int)( y1 * 100 );
				New->U.Text.size = DEFAULT_SIZE_TEXT;
				New->U.Text.type = 0;

				ReadDelimitedText(Buffer, Line+4, 255 );
				New->U.Text.Text = strdup(Buffer);
				/* Correction du cadrage: EESCHEMA utilise le centre du texte */
				New->U.Text.x += (New->U.Text.size * strlen(New->U.Text.Text))/2;
				New->U.Text.y += New->U.Text.size / 2;

				New->Unit = 0; New->Convert = CurrentConvert;
				break;

			case 'E': return(1);
				break;
			 }
		}

	return(1);
}


	/*****************************************************************/
	/* int ReadPinDescr( LibraryEntryStruct * LibEntry, char * Line) */
	/*****************************************************************/

int ReadPinDescr( LibraryEntryStruct * LibEntry, char * Line)
{
LibraryDrawEntryStruct *New, *Pin = NULL;
char Text[256], Buff[256], * ptr;
int IsShape, lpin = 300, delta=100, ii, jj;
int NbParts = PartPerPack;

	if( NbParts == 0 ) NbParts = 1;

	for ( ii = 0; ii < NbParts ; ii ++ )
		{
		New = (LibraryDrawEntryStruct *)
						MyZMalloc(sizeof(LibraryDrawEntryStruct));

		New->DrawType = PIN_DRAW_TYPE;
		New->Pnext = LibEntry->Drawings;
		LibEntry->Drawings = New;

		strncpy((char*)&New->U.Pin.Num, "~", 4);
		delta = 100 * atoi(Line+1); lpin = 300;

		New->U.Pin.SizeNum = SIZE_PIN_TEXT;
		New->U.Pin.SizeName = SIZE_PIN_TEXT;
		New->Unit = PartPerPack-ii;	/* Les structures sont chainees la derniere creee d'abord */
		New->Convert = CurrentConvert;
		New->U.Pin.PinType = PIN_INPUT;
		New->U.Pin.PinShape = NONE;
		New->U.Pin.Flags = 0;			/* Pin visible */
		}

	Pin = LibEntry->Drawings;

	strcpy(Text, Line);
	ptr = strtok(Text, " \t\n");

	/* Lecture Numero de Pin */
	for( New = Pin, ii = PartPerPack; ii > 0; ii-- , New = New->Pnext)
		{
		ptr = strtok(NULL, " '\t\n");
		if( ptr ) strncpy((char*)&New->U.Pin.Num, ptr, 4);
		}

	/* Lecture forme ( si elle existe ) */
	IsShape = 0;
	ptr = strtok(NULL, " \t\n");
	New = Pin;
	if(ptr)
		{
		if( strcmp ( ptr, "DOT") == 0 )
			{
			New->U.Pin.PinShape = INVERT; IsShape = 1;
			}
		if( strcmp ( ptr, "CLK") == 0 )
			{
			New->U.Pin.PinShape |= CLOCK; IsShape = 1;
			}

		if( strcmp ( ptr, "SHORT") == 0 )
			{
			lpin = 100;  IsShape = 1;
			}

		}

	/* Lecture type */
	if ( IsShape ) ptr = strtok(NULL, " \t\n");

	if( ptr) if( strcmp ( ptr, "CLK") == 0 )
		{
		ptr = strtok(NULL, " \t\n");
		New->U.Pin.PinShape |= CLOCK;
		}

	if( ptr) if( strcmp ( ptr, "PAS") == 0 )
		New->U.Pin.PinType = PIN_PASSIVE;
	if( ptr) if( strcmp ( ptr, "OUT") == 0 )
		New->U.Pin.PinType = PIN_OUTPUT;
	if( ptr) if( stricmp ( ptr, "hiZ") == 0 )
		New->U.Pin.PinType = PIN_TRISTATE;
	if( ptr) if( strcmp ( ptr, "OC") == 0 )
		New->U.Pin.PinType = PIN_OPENCOLLECTOR;
	if( ptr) if( strcmp ( ptr, "I/O") == 0 )
		New->U.Pin.PinType = PIN_BIDI;
	if( ptr) if( strcmp ( ptr, "PWR") == 0 )
		{
		New->U.Pin.PinType = PIN_POWER;
		New->U.Pin.Flags = 1;			 /* Pin invisible */
		}

	/* Lecture name */
	ptr = strtok(NULL, " \t\n");
	if( ptr)
		{
		ReadDelimitedText(Buff, ptr, 255);
		/* Suppression des '\' */
		if (Buff[0] == '\\' ) Buff[0] = '~';
		for( ii = 1, jj = 1; ; ii++, jj++ )
			{
			if(Buff[jj] == '\\') jj++;
			Buff[ii] = Buff[jj];
			if(Buff[ii] == 0) break;
			}

		ChangeSpaces(Buff, '_');
		New->U.Pin.Name = strdup(Buff);

		if( PartPerPack == 0)
			{
			strncpy((char*)&New->U.Pin.Num, Buff, 4);
			}
		}

	if( New->U.Pin.Flags == 1) /* Pin invisible -> mise a longueur nulle */
		{
		lpin = 0;
		}
	New->U.Pin.Len = lpin;

	if( Line[0] == 'L' )
		{
		New->U.Pin.Orient = PIN_RIGHT;
		New->U.Pin.posX = LibEntry->BBoxMinX - lpin;
		New->U.Pin.posY = LibEntry->BBoxMaxY - delta;
		}
 
	if( Line[0] == 'R' )
		{
		New->U.Pin.posX = LibEntry->BBoxMaxX + lpin;
		New->U.Pin.posY = LibEntry->BBoxMaxY - delta;
		New->U.Pin.Orient = PIN_LEFT;
		}

	if( Line[0] == 'T' )
		{
		New->U.Pin.posX = LibEntry->BBoxMinX + delta;
		New->U.Pin.posY = LibEntry->BBoxMaxY + lpin;
		New->U.Pin.Orient = PIN_DOWN;
		}
 
	if( Line[0] == 'B' )
		{
		New->U.Pin.posX = LibEntry->BBoxMinX + delta;
		New->U.Pin.posY = LibEntry->BBoxMinY - lpin;
		New->U.Pin.Orient = PIN_UP;
		}

	/* Copie pour les autres Pins */

	for( New = Pin->Pnext, ii = PartPerPack-1; ii > 0; ii-- , New = New->Pnext)
		{
		New->U.Pin.Len = Pin->U.Pin.Len;
		New->U.Pin.Orient = Pin->U.Pin.Orient;
		New->U.Pin.posX = Pin->U.Pin.posX;
		New->U.Pin.posY = Pin->U.Pin.posY;
		New->U.Pin.Name = strdup(Pin->U.Pin.Name);
		New->U.Pin.PinType = Pin->U.Pin.PinType;
		New->U.Pin.Flags = Pin->U.Pin.Flags;
		}

	return(1);
}


	/**************************************************************/
	/* LibraryEntryStruct *FindLibPart(char *Name, int Alias = 0) */
	/**************************************************************/

/*
 Routine to find a part in CurrentLib.
	Name = Name of part.
	Alias = Flag: si flag != 0, retourne un pointeur sur une part ou un alias
				  si flag = 0, retourne un pointeur sur une part meme si le nom
				  correspond a un alias
*/

static LibraryEntryStruct *FindLibPart(char *Name,  int Alias)
{
LibraryEntryStruct *Entry , DummyEntry;
LibraryStruct *Lib = &CurrentLib;

	DummyEntry.Drawings = NULL;	  /* Used only to call PQFind. */
	strcpy(DummyEntry.Name, Name);

	PQCompFunc((PQCompFuncType) LibraryEntryCompare);

	Entry = (LibraryEntryStruct*)PQFind(Lib->Entries, &DummyEntry);

	/* Si le nom est un alias, recherche du vrai composant */
	if( Entry )
		{
		if( (Entry->Type != ROOT ) && (Alias == FIND_ROOT) )
			Entry = FindLibPart( ((LibraryAliasStruct*)Entry)->RootName, FIND_ROOT);
		}
	return (Entry);
}

	/***********************************************/
	/* void ChangeSpaces(char * Text, int NewChar) */
	/***********************************************/

/* Change dans un texte les espaces en NewChar */
static void ChangeSpaces(char * Text, int NewChar)
{
	while( *Text )
		{
		if( *Text == ' ') *Text = (char) NewChar;
		Text++;
		}
}


