/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.daemon;

import java.util.Iterator;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jrobin.core.Util;
import org.videolan.vlma.common.IVlData;
import org.videolan.vlma.common.VlServer;

/**
 * This class is the daemon which monitores servers. It monitores their 
 * state using SNMP and creates RRD graphs.
 * 
 * @author SylV
 * @version 1.0
 */
public class VlServerMonitoring {

    private static final Logger logger = Logger.getLogger(VlServerMonitoring.class);

    private IVlData data;

    /**
     * The time interval between two interrogation requests.
     */
    public static final int TIME_INTERVAL = 300;

    private Thread checkVLCThread;

    private Thread updateSNMPDataThread;

    private Thread serverMonitorDeamonThread;

    public boolean isCheckingVLC() {
        return (checkVLCThread != null) && (checkVLCThread.isAlive());
    };

    public boolean isUpdatingSNMPData() {
        return (updateSNMPDataThread != null)
                && (updateSNMPDataThread.isAlive());
    };

    public boolean isDeamonMonitoring() {
        return (serverMonitorDeamonThread != null)
                && (serverMonitorDeamonThread.isAlive());
    };

    /**
     * This thread try to connect to the VLCs of all servers to check if they
     * run correctly or not. 
     */
    private Runnable vLCchecker = new Runnable() {
        public void run() {
            Iterator i = data.getServers().iterator();
            while (i.hasNext()) {
                VlServer server = (VlServer) i.next();
                server.checkVLC();
            }
        }
    };

    /**
     * This method starts the CheckVLCThread
     */
    synchronized public void startCheckVLCThread() {
        if (!isCheckingVLC()) {
            checkVLCThread = new Thread(vLCchecker);
            checkVLCThread.setName("checkVLCThread");
            checkVLCThread.start();
        }
    }

    /**
     * This method is the main thread which monitores servers. At regular 
     * intervals, servers are sent SNMP requests and RRD graphs are updated.
     */
    Runnable snmpDataUpdater = new Runnable() {
        public void run() {
            Iterator i = data.getServers().iterator();
            //logger.log(Level.DEBUG, "Test");
            while (i.hasNext()) {
                VlServer server = (VlServer) i.next();
                logger.log(Level.DEBUG, "Mise à jour RRD de " + server.getName());
                server.updateSnmpData();
                server.updateRrdGraph();
            }
        }
    };

    /**
     * This method starts the SNMP data updater in a thread
     */
    synchronized public void startUpdateSNMPDataThread() {
        if (!isUpdatingSNMPData()) {
            updateSNMPDataThread = new Thread(snmpDataUpdater);
            updateSNMPDataThread.setName("updateSNMPDataThread");
            updateSNMPDataThread.start();
        }
    }

    /**
     * This thread is the main thread which starts the VLC checker and the SNMP
     * data updater once before waiting and then looping.
     */
    Runnable serverMonitorDeamon = new Runnable() {
        public void run() {
            while (true) {
                logger.log(Level.DEBUG, "Test");
                startCheckVLCThread(); //The VLC thread checker is launched
                startUpdateSNMPDataThread(); //The SNMP Data updater thread checker is launched
                try { //Then we wait before looping
                    Thread.sleep(1000 * (TIME_INTERVAL - (Util.getTime() % TIME_INTERVAL)));
                } catch (InterruptedException e) {
                    //Do nothing
                }
            }
        }
    };

    /**
     * This method is the constructor of the class.
     * It needs the data object of the servers it will check and extract SNMP
     * data.
     * 
     * @param data
     * The data object whose servers will be controlled.
     */

    public synchronized void startServerMonitoringDeamon() {
        if (!isDeamonMonitoring()) {
            serverMonitorDeamonThread = new Thread(serverMonitorDeamon);
            serverMonitorDeamonThread.setName("serverMonitorThread");
            serverMonitorDeamonThread.start();
        }
    }

    public VlServerMonitoring(IVlData data) {
        this.data = data;
    }

}
