/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.monitor;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.log4j.Logger;
import org.jrobin.core.Util;
import org.videolan.vlma.VLMaService;
import org.videolan.vlma.model.Server;

/**
 * This class is the daemon which monitores servers. It monitores their
 * state using SNMP and creates RRD graphs.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 */
public class ServerMonitor implements Monitor {

    private static final Logger logger = Logger.getLogger(ServerMonitor.class);

    private VLMaService vlmaService;

    /**
     * The time interval between two interrogation requests.
     */
    public static final int TIME_INTERVAL = 300;

    private Thread updateSNMPDataThread;

    private Thread serverMonitorDaemonThread;

    private AtomicBoolean shouldRun = new AtomicBoolean(false);

    public boolean isUpdatingSNMPData() {
        return (updateSNMPDataThread != null)
                && (updateSNMPDataThread.isAlive());
    }

    public boolean isDaemonMonitoring() {
        return (serverMonitorDaemonThread != null)
                && (serverMonitorDaemonThread.isAlive());
    }

    /**
     * This method is the main thread which monitores servers. At regular
     * intervals, servers are sent SNMP requests and RRD graphs are updated.
     */
    Runnable snmpDataUpdater = new Runnable() {
        public void run() {
            synchronized(ServerMonitor.this) {
                List<Thread> updateThreads = new ArrayList<Thread>();
                for (Server server : vlmaService.getServers()) {
                    Thread updateThread = new UpdateThread(vlmaService, server);
                    updateThreads.add(updateThread);
                    updateThread.start();
                }
                for (Thread updateThread : updateThreads) {
                    try {
                        updateThread.join();
                    } catch (InterruptedException e) { }
                }
            }
        }
    };

    /**
     * This method starts the SNMP data updater in a thread
     */
    synchronized public void startUpdateSNMPDataThread() {
        if (!isUpdatingSNMPData()) {
            updateSNMPDataThread = new Thread(snmpDataUpdater);
            updateSNMPDataThread.setName("updateSNMPDataThread");
            updateSNMPDataThread.start();
        }
    }

    /**
     * This thread is the main thread which starts the VLC checker and the SNMP
     * data updater once before waiting and then looping.
     */
    Runnable serverMonitorDaemon = new Runnable() {
        public void run() {
            shouldRun.set(true);
            while (shouldRun.get()) {
                try {
                    // Wait before looping
                    Thread.sleep(1000 * (TIME_INTERVAL - (Util.getTime() % TIME_INTERVAL)));
                } catch (InterruptedException e) {
                    // Do nothing
                }
                if (shouldRun.get())
                    startUpdateSNMPDataThread();
            }
            logger.debug("ServerMonitor thread stopped.");
        }
    };


    public synchronized void start() {
        if (!isDaemonMonitoring()) {
            logger.info("Starting " + this.getClass().getSimpleName());
            serverMonitorDaemonThread = new Thread(serverMonitorDaemon);
            serverMonitorDaemonThread.setName("serverMonitorThread");
            serverMonitorDaemonThread.start();
        }
    }

    public synchronized void stop() {
        logger.info("Stopping " + this.getClass().getSimpleName());
        shouldRun.set(false);
        serverMonitorDaemonThread.interrupt();
    }

    /**
     * Sets the VLMa service.
     *
     * @param vlmaService the vlmaService to set
     */
    public void setVlmaService(VLMaService vlmaService) {
        this.vlmaService = vlmaService;
    }

    private class UpdateThread extends Thread {

        private VLMaService vlmaService;
        private Server server;

        public UpdateThread(VLMaService vlmaService, Server server) {
            this.vlmaService = vlmaService;
            this.server = server;
        }

        public void run() {
            logger.debug("Update RRD file of " + server.getName());
            vlmaService.updateSnmpData(server);
            vlmaService.updateRrdGraph(server);
        }
    }
}
