/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.monitor;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;
import org.videolan.vlma.VLMaService;
import org.videolan.vlma.model.Server;

/**
 * This class is the daemon which monitores the servers' state. It periodically
 * tries to connect to the telnet interface to check whether VLC is up and
 * running.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 * @author Adrien Grand <jpountz at videolan.org>
 */
public class ServerStateMonitor implements Monitor {

    private static final Logger logger = Logger.getLogger(ServerStateMonitor.class);

    private Configuration configuration;

    private VLMaService vlmaService;

    private Thread checkVLCThread;

    private Thread serverStateMonitorDaemonThread;

    private AtomicBoolean shouldRun = new AtomicBoolean(true);

    public boolean isCheckingVLC() {
        return (checkVLCThread != null) && (checkVLCThread.isAlive());
    }

    public boolean isDaemonMonitoring() {
        return (serverStateMonitorDaemonThread != null)
                && (serverStateMonitorDaemonThread.isAlive());
    }

    /**
     * This thread try to connect to the VLCs of all servers to check if they
     * run correctly or not.
     */
    private Runnable vLCchecker = new Runnable() {
        public void run() {
            List<Thread> checkThreads = new ArrayList<Thread>();
            for (Server server : vlmaService.getServers()) {
                Thread checkThread = new CheckThread(vlmaService, server);
                checkThreads.add(checkThread);
                checkThread.start();
            }
            for(Thread checkThread : checkThreads) {
                try {
                    checkThread.join();
                } catch (InterruptedException e) { }
            }
        }
    };

    /**
     * This method starts the CheckVLCThread
     */
    synchronized public void startCheckVLCThread() {
        if (!isCheckingVLC()) {
            checkVLCThread = new Thread(vLCchecker);
            checkVLCThread.setName("checkVLCThread");
            checkVLCThread.start();
        }
    }

    /**
     * This thread is the main thread which starts the VLC checker and the SNMP
     * data updater once before waiting and then looping.
     */
    Runnable serverStateMonitor = new Runnable() {
        public void run() {
            shouldRun.set(true);
            while (shouldRun.get()) {
                try {
                    // Wait before looping
                    Thread.sleep(1000 * configuration.getInt("vlma.monitor.server.interval"));
                } catch (InterruptedException e) {
                    //Do nothing
                }
                if (shouldRun.get())
                    startCheckVLCThread();
            }
            logger.debug("ServerStateMonitor thread stopped.");
        }
    };


    public synchronized void start() {
        if (!isDaemonMonitoring()) {
            logger.info("Starting " + this.getClass().getSimpleName());
            serverStateMonitorDaemonThread = new Thread(serverStateMonitor);
            serverStateMonitorDaemonThread.setName("serverStateMonitorThread");
            serverStateMonitorDaemonThread.start();
        }
    }

    public synchronized void stop() {
        logger.info("Stopping " + this.getClass().getSimpleName());
        shouldRun.set(false);
        serverStateMonitorDaemonThread.interrupt();
    }

    /**
     * Sets the VLMa service.
     *
     * @param vlmaService the vlmaService to set
     */
    public void setVlmaService(VLMaService vlmaService) {
        this.vlmaService = vlmaService;
    }

    /**
     * Sets the configuration.
     *
     * @param configuration the configuration to set
     */
    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    private class CheckThread extends Thread {

        private VLMaService vlmaService;
        private Server server;

        public CheckThread(VLMaService vlmaService, Server server) {
            this.vlmaService = vlmaService;
            this.server = server;
        }

        public void run() {
            vlmaService.checkVLC(server);
        }
    }
}
